/*
 * Full Screen Background v2.0.1 (03 Oct 2012)
 * 
 * Copyright 2011-2012 ThemeCatcher
 * 
 * All rights reserved
 */
(function ($, window) {
    'use strict';
    // Default settings
    var defaults = {
        speed: 2000,                        // Speed of the transition between background images, in milliseconds 1000 = 1 second
        transition: 'fade',                 // The transition animation. 'fade', 'fadeOutFadeIn', 'slideDown', 'slideRight', 'slideUp', 'slideLeft', 'carouselRight', 'carouselLeft'
        position: 'fixed',                  // Whether the background is positioned absolute or fixed
        fitLandscape: false,                // If landscape images should be locked to 100% width
        fitPortrait: true,                  // If portrait images should be locked to 100% height
        fitAlways: false,                   // Don't crop images at all
        positionX: 'center',                // Where to position the image on the X axis. 'left', 'center', 'right'
        positionY: 'center',                // Where to position the image on the Y axis. 'top', 'center', 'bottom'
        easing: 'swing',                    // The easing function to use when transitioning
        controlsSelector: '#fs-controls',   // The jQuery selector of the element to append the controls to
        hideSpeed: 1000,                    // Speed that the website is hidden at, when activating full screen mode, in milliseconds
        showSpeed: 1000,                    // Speed that the website is shown at, when closing full screen mode, in milliseconds
        controlSpeed: 500,                  // Speed that the controls fades in, in full screen mode, in milliseconds
        fadeIE: false,                      // Whether or not to fade the website in IE 7,8
        save: true,                         // Whether or not to save the current background across pages
        slideshow: true,                    // Whether or not to use the slideshow functionality
        slideshowAuto: true,                // Whether or not to start the slideshow automatically
        slideshowSpeed: 4000,               // How long the slideshow stays on one image, in milliseconds
        random: false,                      // Whether the images should be displayed in random order, forces save = false
        keyboard: true,                     // Whether or not to use the keyboard controls, left arrow, right arrow and esc key
        captionPosition: 'center bottom',   // The default caption position
        alwaysShowCaptions: false,          // Always show the captions
        captionSpeed: 600,                  // The speed of the caption fade animation
        bullets: true,                      // Dislay bullet navigation
        lowQuality: false,                  // Turns on lower quality but higher performance transitions
        errorBackground: '',                // Path to the background image shown if there is an error loading an image
        breaker: false,                     // Display breaker image
        breakerOnMax: false,                // Display breaker image in maximise mode
        onInit: false,                      // Callback before the first image is shown
        onLoad: false,                      // Callback when the current image starts loading
        onComplete: false,                  // Callback when the current image has completely loaded
        captionEnhancement: function () {}  // Caption enhancement function e.g. replace Cufon
    },

    // Wrappers & captions
    $outer,
    $stage,
    $captionOuter,
    $caption,
    $breaker,

    // Full screen controls
    $controlsWrap,
    $controls,
    $prev,
    $play,
    $next,
    $loadingWrap,
    $loading,
    $closeWrap,
    $close,
    $bullets,

    // Template background controls
    $fsControls,
    $fsPrev,
    $fsPlay,
    $fsNext,
    $fsMax,

    // Window & body
    $window,
    $body,

    // Misc
    isIE = $.browser.msie && !$.support.opacity,
    cookiePlugin,
    backgrounds,
    total,
    imageCache = [],
    bodyOverflow,
    currentIndex = 0,
    animating = false,
    settings,
    fullscreen,
    captionXPositions = ['left', 'center', 'right'],
    captionYPositions = ['top', 'center', 'bottom'],
    isMaximised = false,
    slideshowTimeout,
    slideshowStarted = false;

    /**
     * Load the image with the given index
     * 
     * @param int index The index of the image to load.
     * @param function callback Callback function to call when loading is finished.
     */
    function load(index, callback) {
        if (backgrounds[index] !== undefined && imageCache[index] === undefined) {
            imageCache[index] = true;
            
            var $image = $('<img/>'),
            errorLoading = false;
            
            $image.load(function () {
                $image.unbind('load');
                
                setTimeout(function () {
                    setImageData($image);
                    resizeImages();
                    
                    if (typeof callback === 'function') {
                        callback();
                    }
                }, 1); // Chrome will sometimes report a 0 by 0 size if there isn't pause in execution
            })
            .error(function () {
                if (!errorLoading && settings.errorBackground) {
                    $image.attr('src', settings.errorBackground);
                }
                errorLoading = true;
            });
            
            setTimeout(function () { 
                $image.attr('src', backgrounds[index].image);
                $('div', $stage).eq(index).append($image);
            }, 1); // Opera 10.6+ will sometimes load the src before the onload function is set, so wait 1ms
        }
    }

    /**
     * Get the index of the next image
     * 
     * @return int
     */
    function getNextIndex() {
        return (currentIndex === (total - 1)) ? 0 : currentIndex + 1;
    }
        
    /**
     * Get the index of the prev image
     * 
     * @return int
     */
    function getPrevIndex() {
        return (currentIndex === 0) ? total - 1 : currentIndex - 1;
    }
        
    /**
     * Return a random value from the given array
     * 
     * @param array array
     * @return array
     */
    function random(array) {
        return array[Math.floor(Math.random() * array.length)];
    }
    
    /**
     * Randomly shuffle a given array
     * 
     * @param array array
     * @return array
     */
    function shuffle(array)
    {
        var tmp, current, top = array.length;

        if(top) {
            while(--top) {
                current = Math.floor(Math.random() * (top + 1));
                tmp = array[current];
                array[current] = array[top];
                array[top] = tmp;
            }
        }

        return array;
    }
    
    /**
     * Trigger the given event and call callback
     * 
     * @param string event
     * @param function callback
     */
    function trigger(event, callback) {
        if (typeof callback === 'function') {
            callback.call();
        }
        $.event.trigger(event);
    }
    
    /**
     * Get a cookie
     * 
     * @param string key The key of the cookie.
     */
    function getCookie(key)
    {
        if (cookiePlugin) {
            return $.cookie(key);
        }
    }
    
    /**
     * Set a cookie
     * 
     * @param string key The key of the cookie.
     * @param string value The value of the cookie.
     * @param object options An object literal containing key/value pairs to provide optional cookie attributes.
     */
    function setCookie(key, value, options) {
        if (cookiePlugin) {
            $.cookie(key, value, options);
        }
    }
    
    fullscreen = $.fullscreen = function (options) {
        settings = $.extend({}, defaults, options || {});
        
        backgrounds = settings.backgrounds;
        total = backgrounds.length;
        
        if (!total) {
            return;
        }
        
        for (var i = 0; i < total; i++) {
            if (typeof backgrounds[i] === 'string') {
                backgrounds[i] = { image: backgrounds[i] };
            }
        }
        
        if (settings.random) {
            backgrounds = shuffle(backgrounds);
            settings.save = false;
        }

        if (typeof settings.backgroundIndex === 'number') {
            currentIndex = settings.backgroundIndex;
            settings.save = false;
        }
        
        if (isIE && !settings.fadeIE) {
            settings.hideSpeed = 0;
            settings.showSpeed = 0;
            settings.controlSpeed = 0;
        }
                         
        init();
    };
    
    /**
     * Intialisation
     * 
     * Sets up the HTML elements and JavaScript bindings then loads
     * the first image.
     */
    function init() {
        // Cache some common vars
        $window = $(window);
        $body = $('body');
        cookiePlugin = !!$.cookie;
        
        // Create the div structure
        $outer = $('<div class="fullscreen-outer"></div>').append(
            $stage = $('<div class="fullscreen-stage"></div>'),
            $breaker = $('<div class="fullscreen-breaker"/>'),
            $captionOuter = $('<div class="fullscreen-caption-outer"></div>').append(
                $('<div class="fullscreen-caption-helper"></div>').append(
                    $caption = $('<div class="fullscreen-caption"></div>')
                )
            )
        );
        
        $controlsWrap = $('<div class="fullscreen-controls-outer"></div>').append(
            $controls = $('<div class="fullscreen-controls"></div>').append(
                $prev = $('<div class="fullscreen-prev"></div>'),
                $play = $('<div class="fullscreen-play"></div>'),
                $next = $('<div class="fullscreen-next"></div>')
            ),
            $loadingWrap = $('<div class="fullscreen-loading-wrap"></div>').append(
                $loading = $('<div class="fullscreen-loading"></div>')
            ),
            $closeWrap = $('<div class="fullscreen-close-wrap"></div>').append(
                $close = $('<div class="fullscreen-close"></div>')
            )
        );
        
        $fsControls = $('<div class="fs-controls"></div>').append(
            $fsPrev = $('<div class="fs-prev"></div>'),
            $fsPlay = $('<div class="fs-play"></div>'),
            $fsNext = $('<div class="fs-next"></div>'),
            $fsMax = $('<div class="fs-max"></div>')
        );
                
        for (var i = 0; i < total; i++) {
            $stage.append('<div class="fullscreen-slide"/>');
        }
                
        if (settings.position === 'absolute') {
            $body.addClass('fs-absolute');
        } else {
            $body.addClass('fs-fixed');
        }
                
        // Put the controls on the page
        if (settings.controlsSelector) {
            $(settings.controlsSelector).append($fsControls);
        }
                
        $body.addClass('fullscreen-background fullscreen-mode-normal').prepend($outer).append($controlsWrap);
               
        if (total > 1) {
            $controls.add($fsPrev).add($fsNext).show();
            
            // Bullets functionality
            if (settings.bullets) {
                $bullets = $('<div class="fullscreen-bullets"/>');
                
                for (var j = 0; j < total; j++) {
                    var title = backgrounds[j].title || '';
                    $('<div class="fullscreen-bullet">' + j + '</div>').data('index', j).attr('title', title).appendTo($bullets);
                }
                
                $('.fullscreen-bullet', $bullets).click(function () {
                    fullscreen.go($(this).data('index'));
                    return false;
                });
                
                $controls.append($bullets);
            }
            
            // Slideshow functionality
            if (settings.slideshow) {
                if (getCookie('fullscreenSlideshow') === 'start') {
                    fullscreen.start();
                } else if (getCookie('fullscreenSlideshow') === 'stop') {
                    fullscreen.stop();
                } else {
                    if (settings.slideshowAuto) {
                        fullscreen.start();
                    } else {
                        fullscreen.stop();
                    }
                }
                
                $play.add($fsPlay).show();
                $controlsWrap.add($fsControls).addClass('fs-slideshow');               
            }
        } else {
            $controlsWrap.add($fsControls).addClass('fs-single');
            settings.slideshow = false;
        }
                
        // Bind the prev button to load the previous image
        $prev.add($fsPrev).click(function () {         
            fullscreen.prev();
            return false;
        });
        
        // Bind the next button to load the next image
        $next.add($fsNext).click(function () {
            fullscreen.next();
            return false;
        });
        
        // Bind the maximise buttons to enter maximise mode
        $fsMax.click(fullscreen.max);
        
        // Bind the close button to close it
        $closeWrap.click(fullscreen.close);
        
        // Save the current body overflow value
        bodyOverflow = $body.css('overflow');

        if (settings.save) {
            // Check for the saved background cookie to override the default
            var savedBackground = parseInt(getCookie('fullscreenSavedBackground'), 10);
            for(var k = 0; k < total; k++) {
                if (k === savedBackground) {
                    currentIndex = k;
                    break;
                }
            }
        }
                
        trigger('fullscreenInit', settings.onInit);
        
        // Preload the previous image
        if (total > 2) {
            load(getPrevIndex());
        }
        
        // Load the current image then do the first transition
        var loadingTimeout = setTimeout(function () { $controlsWrap.add($fsControls).addClass('fs-load'); }, 200);
        
        load(currentIndex, function () {
            clearTimeout(loadingTimeout);
            $controlsWrap.add($fsControls).removeClass('fs-load');
            
            // Put the breaker on the page
            if (settings.breaker) {        
                $breaker.fadeIn('slow');
            }
            
            // Bind the resize funtion when the window is resized
            $window.resize($.isFunction($.throttle) ? $.throttle(100, resizeImages) : resizeImages);
            
            // Do the first transition
            doTransition();
        });
        
        // Preload the next image
        if (total > 1) {
            load(getNextIndex());
        }
    }    
    
    /**
     * Resize the images
     * 
     * @param function callback Called after the resize is complete
     */
    function resizeImages() {
        var windowWidth = $stage.width(),
        windowHeight = $stage.height(),
        windowRatio = windowHeight / windowWidth;
        
        $('img', $stage).each(function () {
            var $image = $(this),
            imageRatio = $image.data('imageRatio'),            
            css = {};
            
            if (windowRatio > imageRatio) {
                // Window is more "portrait" than the image
                if (settings.fitAlways) {
                    $image.width(windowWidth).height(windowWidth * imageRatio);
                } else {
                    if (imageRatio <= 1 && settings.fitLandscape) {
                        $image.width(windowWidth).height(windowWidth * imageRatio);
                    } else {
                        $image.height(windowHeight).width(windowHeight / imageRatio);
                    }
                }
            } else {
                // Window is more "landscape" than the image
                if (settings.fitAlways) {
                    $image.height(windowHeight).width(windowHeight / imageRatio);
                } else {
                    if (imageRatio > 1 && settings.fitPortrait) {
                        $image.height(windowHeight).width(windowHeight / imageRatio);
                    } else {
                        $image.width(windowWidth).height(windowWidth * imageRatio);
                    }
                }
            }
            
            switch (settings.positionX) {               
                case 'left':
                    css.left = 0;
                    css.right = 'auto';
                    break;
                case 'right':
                    css.left = 'auto';
                    css.right = 0;
                    break;
                default:
                case 'center':                
                    css.left = ((windowWidth - $image.width()) / 2) + 'px';
                    css.right = 'auto';
                    break;
            }
            
            switch (settings.positionY) {
                case 'top':
                    css.top = 0;
                    css.bottom = 'auto';
                    break;
                case 'bottom':
                    css.top = 'auto';
                    css.bottom = 0;
                    break;
                default:
                case 'center':
                    css.top = ((windowHeight - $image.height()) / 2) + 'px';
                    css.bottom = 'auto';
                    break;                  
            }
                        
            $image.css(css);
        });
    }
    
    /**
     * Save the image data to use later
     * 
     * @param object $image The jQuery object of the image.
     */
    function setImageData($image) {
        var imageWidth = $image.width(),
        imageHeight = $image.height();
        
        $image.data({
            imageWidth: imageWidth,
            imageHeight: imageHeight,
            imageRatio: imageHeight / imageWidth
        });
    }    
    
    /**
     * Do the transtion animation
     * 
     * @param boolean reverse Reverse the transition animation.
     */
    function doTransition (reverse) {
        trigger('fullscreenLoad', settings.onLoad);
        animating = true;
        
        $controlsWrap.add($fsControls).addClass('fs-animating');
        $('.fs-prev-slide', $stage).removeClass('fs-prev-slide');
        
        var $currentSlide = $('.fs-current-slide', $stage).removeClass('fs-current-slide').addClass('fs-prev-slide'),
        $nextSlide = $('div:eq(' + currentIndex + ')', $stage);
        
        // Preload the next image in the direction we are going
        if (!reverse) {
            load(getNextIndex());               
        } else {
            load(getPrevIndex());
        }
                
        setActiveBullet();
        
        if (settings.save) {
            setCookie('fullscreenSavedBackground', currentIndex, {expires: 365});
        }

        // Hide captions before transitioning
        if (settings.alwaysShowCaptions || isMaximised) {
            $captionOuter.stop().animate({opacity: 0}, settings.captionSpeed, function () {
                $captionOuter.hide();
            });             
        }
        
        $nextSlide.css('visibility', 'hidden').addClass('fs-current-slide');
        
        if (settings.lowQuality) {
            $stage.addClass('fullscreen-low');
        }

        switch (settings.transition) {
            case 'none':
                $nextSlide.css('visibility', 'visible'); doneTransition();
                break;
            default:
            case 'fade':            
                $nextSlide.animate({ opacity: 0 }, 0).css('visibility', 'visible').animate({ opacity: 1 }, settings.speed, settings.easing, doneTransition);                    
                break;
            case 'fadeOutFadeIn':
                var fadeIn = function () {
                    $nextSlide.animate({ opacity: 0 }, 0).css('visibility', 'visible').animate({ opacity: 1 }, settings.speed, settings.easing, doneTransition);
                };
                
                if ($currentSlide.length) {
                    $currentSlide.animate({opacity: 0}, settings.speed, settings.easing, fadeIn);
                } else {
                    fadeIn();
                }
                break;
            case 'slideDown':
                if (!reverse) {
                    $nextSlide.animate({ top: -$stage.height() }, 0).css('visibility', 'visible').animate({ top: 0 }, settings.speed, settings.easing, doneTransition);
                } else {
                    $nextSlide.animate({ top: $stage.height() }, 0).css('visibility', 'visible').animate({ top: 0 }, settings.speed, settings.easing, doneTransition);
                }
                break;
            case 'slideRight':
                if (!reverse) {
                    $nextSlide.animate({ left: $stage.width() }, 0).css('visibility', 'visible').animate({ left: 0 }, settings.speed, settings.easing, doneTransition);
                } else {
                    $nextSlide.animate({ left: -$stage.width() }, 0).css('visibility', 'visible').animate({ left: 0 }, settings.speed, settings.easing, doneTransition);
                }
                break;               
            case 'slideUp':
                if (!reverse) {
                    $nextSlide.animate({ top: $stage.height() }, 0).css('visibility', 'visible').animate({ top: 0 }, settings.speed, settings.easing, doneTransition);
                } else {
                    $nextSlide.animate({ top: -$stage.height() }, 0).css('visibility', 'visible').animate({ top: 0 }, settings.speed, settings.easing, doneTransition);
                }
                break;
            case 'slideLeft':
                if (!reverse) {
                    $nextSlide.animate({ left: -$stage.width() }, 0).css('visibility', 'visible').animate({ left: 0 }, settings.speed, settings.easing, doneTransition);
                } else {
                    $nextSlide.animate({ left: $stage.width() }, 0).css('visibility', 'visible').animate({ left: 0 }, settings.speed, settings.easing, doneTransition);
                }
                break;
            case 'carouselRight':
                if (!reverse) {
                    $nextSlide.animate({ left: $stage.width() }, 0).css('visibility', 'visible').animate({ left: 0 }, settings.speed, settings.easing, doneTransition);
                    $currentSlide.animate({ left: -$stage.width() }, settings.speed, settings.easing);
                } else {
                    $nextSlide.animate({ left: -$stage.width() }, 0).css('visibility', 'visible').animate({ left: 0 }, settings.speed, settings.easing, doneTransition);
                    $currentSlide.animate({ left: 0 }, 0).animate({left: $stage.width()}, settings.speed, settings.easing);
                }
                break; 
            case 'carouselLeft':
                if (!reverse) {
                    $nextSlide.animate({ left: -$stage.width() }, 0).css('visibility', 'visible').animate({ left: 0 }, settings.speed, settings.easing, doneTransition);
                    $currentSlide.animate({left: $stage.width()}, settings.speed, settings.easing);
                } else {
                    $nextSlide.animate({ left: $stage.width() }, 0).css('visibility', 'visible').animate({ left: 0 }, settings.speed, settings.easing, doneTransition);
                    $currentSlide.animate({ left: 0 }, 0).animate({left: -$stage.width()}, settings.speed, settings.easing);
                }
                break; 
        }        
    }
    
    /**
     * Actions to run when the transition animation is complete
     */
    function doneTransition() {        
        animating = false;
                
        if (settings.lowQuality) {
            $stage.removeClass('fullscreen-low');
        }        
        
        var caption = backgrounds[currentIndex].caption || '',
        captionPosition = backgrounds[currentIndex].captionPosition || settings.captionPosition;
        
        if (captionPosition === 'random') {
            captionPosition = random(captionXPositions) + ' ' + random(captionYPositions);
        }
        
        if (caption) {
            $caption.html(caption);
            settings.captionEnhancement.call($captionOuter);
            $captionOuter.attr('class', 'fullscreen-caption-outer') // Remove any previous caption position class
                         .addClass('fs-position-' + captionPosition.split(' ').join('-'));
            
            if (settings.alwaysShowCaptions || isMaximised) {
                $captionOuter.stop(true, true).show().animate({opacity: 1}, settings.captionSpeed);
            }
        } else {
            $caption.html('');
        }
        
        $controlsWrap.add($fsControls).removeClass('fs-animating');
        
        trigger('fullscreenComplete', settings.onComplete);
    }
    
    /**
     * Set the active bullet
     */
    function setActiveBullet() {
        if (settings.bullets && total > 1) {
            $bullets.children().removeClass('active-bullet').eq(currentIndex).addClass('active-bullet');
        }
    }
    
    /**
     * Go to full screen mode
     */
    fullscreen.max = function () {
        $body.css('overflow', 'hidden');
        
        if (!settings.breakerOnMax) {
            $breaker.fadeOut(settings.hideSpeed);
        }
        
        $('#outside').fadeOut(settings.hideSpeed).hide(0, function () {
            isMaximised = true;
            $body.removeClass('fullscreen-mode-normal').addClass('fullscreen-mode-full');
            $controlsWrap.fadeIn(settings.controlSpeed).show(0, function () {
                if (settings.keyboard) {
                    $(document).bind('keydown.fullscreen', function (e) {
                        if (e.keyCode === 27) {
                            e.preventDefault();
                            fullscreen.close();
                        }
                    });
                }
            });
            
            if(!animating && $caption.html()) {
                $captionOuter.show().animate({opacity: 1}, settings.captionSpeed);
            }
            
            if (settings.keyboard) {
                $(document).bind('keydown.fullscreen', function (e) {
                    if (e.keyCode === 37) {
                        e.preventDefault();
                        fullscreen.prev();
                    } else if (e.keyCode === 39) {
                        e.preventDefault();
                        fullscreen.next();
                    }
                });
            }
        });
        
        $window.resize();
    };
    
    /**
     * Exit from full screen mode
     */
    fullscreen.close = function () {
        $(document).unbind('keydown.fullscreen');
        $controlsWrap.stop(true, true).hide();
        if (!settings.alwaysShowCaptions) {
            $captionOuter.stop(true, true).hide().css({ opacity: 0 });
        }
        isMaximised = false;
        
        $('#outside').fadeIn(settings.showSpeed);
        
        if (settings.breaker) {
            $breaker.fadeIn(settings.showSpeed);
        }
                
        $body.removeClass('fullscreen-mode-full').addClass('fullscreen-mode-normal').css('overflow', bodyOverflow);
        
        $window.resize();
    };
    
    /**
     * Load the next image
     */
    fullscreen.next = function () {
        if (animating) {
            return false;
        }
        
        currentIndex = getNextIndex();            
        doTransition();
    };
    
    /**
     * Load the previous image
     */
    fullscreen.prev = function () {
        if (animating) {
            return false;
        }
        
        currentIndex = getPrevIndex();    
        doTransition(true);
    };
    
    /**
     * Load the image at the given index
     */
    fullscreen.go = function (index) {
        if (animating || currentIndex === index) {
            return false;
        }        
              
        if (index > currentIndex) {
            currentIndex = index;
            
            // Make sure this image is loaded
            if (imageCache[currentIndex] === undefined) {
                load(currentIndex, doTransition);
            } else {
                doTransition();
            }
        } else {
            currentIndex = index;
            
            // Make sure this image is loaded
            if (imageCache[currentIndex] === undefined) {
                load(currentIndex, function () {
                    doTransition(true);
                });
            } else {
                doTransition();
            }
        }
    };
    
    /**
     * Start the slideshow
     */
    fullscreen.start = function () {
        if (settings.slideshow && !slideshowStarted) {
            slideshowStarted = true;
            
            $play               
                .bind('fullscreenComplete', function () {
                    slideshowTimeout = setTimeout(fullscreen.next, settings.slideshowSpeed);
                })
                .bind('fullscreenLoad', function () {                        
                    clearTimeout(slideshowTimeout);
                })
                .removeClass('fullscreen-play')
                .addClass('fullscreen-pause')
                .add($fsPlay)
                .unbind('click')
                .one('click', function () {
                    setCookie('fullscreenSlideshow', 'stop');
                    fullscreen.stop();
                });
            $fsPlay             
                .removeClass('fs-play')
                .addClass('fs-pause');

            slideshowTimeout = setTimeout(fullscreen.next, settings.slideshowSpeed);
        }
    };
    
    /**
     * Stop the slideshow
     */
    fullscreen.stop = function () {
        if (settings.slideshow) {
            clearTimeout(slideshowTimeout);
            
            $play
                .unbind('fullscreenLoad fullscreenComplete')
                .removeClass('fullscreen-pause')
                .addClass('fullscreen-play')
                .add($fsPlay)
                .unbind('click')
                .one('click', function () {
                    setCookie('fullscreenSlideshow', 'start');
                    fullscreen.start();
                });
            $fsPlay
                .removeClass('fs-pause')
                .addClass('fs-play');
            
            slideshowStarted = false;
        }
    };
})(jQuery, window);

