<?php

// Prevent direct script access
if (!defined('ABSPATH')) exit;

/**
 * Popular posts widget
 */
class TCR_Widget_Popular extends WP_Widget {

	function __construct() {
		$widget_ops = array('classname' => 'widget_tcr_popular', 'description' => __('Display popular posts', TCR_ADMIN_TEXT_DOMAIN));
		$control_ops = array('width' => 400, 'height' => 350);
		parent::__construct('tcr_popular', TCR_NAME . ' - ' . __('Popular Posts', TCR_ADMIN_TEXT_DOMAIN), $widget_ops, $control_ops);

	    add_action('save_post', array(&$this, 'flush_widget_cache'));
		add_action('deleted_post', array(&$this, 'flush_widget_cache'));
		add_action('switch_theme', array(&$this, 'flush_widget_cache'));
	}

	function widget( $args, $instance ) {
	    $cache = wp_cache_get('tcr_widget_popular', 'widgets');

	    if (!is_array($cache)) {
	        $cache = array();
	    } elseif (isset($cache[$args['widget_id']])) {
	        echo $cache[$args['widget_id']];
	        return;
	    }

	    global $tcr, $post;

		extract($args);

		$title = apply_filters('widget_title', $instance['title'], $instance, $this->id_base);
		$instance['show_information'] = $instance['show_information'] == 'both' ? array('date', 'description') : array($instance['show_information']);

		ob_start();

		$query = array(
		    'post__not_in' => array($post->ID),
		    'posts_per_page' => $instance['count'],
		    'post_status' => 'publish',
		    'ignore_sticky_posts' => 1,
		    'orderby' => 'comment_count',
		    'order' => 'DESC'
		);

		if (is_array($instance['categories']) && count($instance['categories'])) {
		    $query['cat'] = join(',', $instance['categories']);
		}

	    if (is_array($instance['authors']) && count($instance['authors'])) {
		    $query['author'] = join(',', $instance['authors']);
		}

	    $items = new WP_Query($query);
	    if ($items->have_posts()) :

    		echo $before_widget;
    		if ($title) { echo $before_title . $title . $after_title; } ?>

    		<div class="widget-posts widget-popular-posts">
    		    <?php while ($items->have_posts()) : $items->the_post(); ?>
                    <div class="widget-post clearfix">
                        <?php if ($instance['show_thumbnail']) : ?>
                            <div class="widget-post-thumb">
                                <a class="widget-post-thumb-link" href="<?php the_permalink(); ?>" title="<?php the_title(); ?>">
                                    <?php if (has_post_thumbnail()) : ?>
                                        <?php the_post_thumbnail(array(65, 65), array('title' => get_the_title(), 'alt' => get_the_title())); ?>
                                    <?php else : ?>
                                        <?php
                                            $src = tcr_get_image_src(TCR_URL . '/images/widget-post-thumb.png', array('width' => 65, 'height' => 65));
                                            echo tcr_get_image_tag($src, array('title' => get_the_title(), 'alt' => get_the_title()));?>
                                    <?php endif; ?>
                                </a>
                            </div>
                        <?php endif; // show thumbnail ?>
                        <div class="widget-post-info">
                            <h4><a href="<?php the_permalink(); ?>" rel="bookmark" title="<?php echo esc_attr(get_the_title()); ?>">
                                <?php
                                    if ($instance['title_length'] && mb_strlen(get_the_title()) > $instance['title_length']) {
                                        echo mb_substr(get_the_title(), 0, $instance['title_length']) . '...';
                                    } else {
                                        the_title();
                                    }
                                ?>
                            </a></h4>
                            <?php if (in_array('date', $instance['show_information'])) : ?>
                                <div class="widget-post-date">
                                    <?php echo get_the_date(); ?>
                                </div>
                            <?php endif; ?>
                            <?php if (in_array('description', $instance['show_information'])) : ?>
                                <p class="widget-post-description">
                                    <?php echo wp_html_excerpt(get_the_excerpt(), $instance['description_length']); ?>
                                </p>
                            <?php endif; ?>
                        </div>
                    </div>
    		    <?php endwhile; // have_posts() ?>
    		</div>

    		<?php echo $after_widget;

		endif; // have_posts()

		$output = ob_get_clean();

		wp_reset_postdata();

		$cache[$args['widget_id']] = $output;
		wp_cache_set('tcr_widget_popular', $cache, 'widgets');

		echo $output;
	}

	function update( $new_instance, $old_instance ) {
		$instance = $old_instance;
		$instance['title'] = strip_tags($new_instance['title']);
		$instance['count'] = absint($new_instance['count']);
        $instance['show_thumbnail'] = isset($new_instance['show_thumbnail']) ? true : false;
		$instance['show_information'] = $new_instance['show_information'];
        $instance['title_length'] = $new_instance['title_length'] ? absint($new_instance['title_length']) : '';
        $instance['description_length'] = $new_instance['description_length'] ? absint($new_instance['description_length']) : '';
		$instance['categories'] = isset($new_instance['categories']) && is_array($new_instance['categories']) ? $new_instance['categories'] : array();
		$instance['authors'] = isset($new_instance['authors']) && is_array($new_instance['authors']) ? $new_instance['authors'] : array();
        return $instance;
	}

	function form( $instance ) {
		$instance = wp_parse_args((array) $instance, array(
		    'title' => tcr__('popular_posts', __('Popular Posts', TCR_TEXT_DOMAIN)),
		    'count' => 5,
		    'show_thumbnail' => true,
		    'show_information' => 'date',
		    'title_length' => 20,
		    'description_length' => 70,
		    'categories' => array(),
		    'authors' => array()
	    ));
?>
        <div class="tcr-widgetfield-wrap">
            <div class="tcr-widgetfield-label">
                <label for="<?php echo $this->get_field_id('title'); ?>"><?php esc_html_e('Title', TCR_ADMIN_TEXT_DOMAIN); ?></label>
            </div>
            <div class="tcr-widgetfield-input">
		        <input class="tcr-width-300" id="<?php echo $this->get_field_id('title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr($instance['title']); ?>" />
            </div>
        </div>

        <div class="tcr-widgetfield-wrap">
            <div class="tcr-widgetfield-label">
                <label for="<?php echo $this->get_field_id('count'); ?>"><?php esc_html_e('Number of posts', TCR_ADMIN_TEXT_DOMAIN); ?></label>
            </div>
            <div class="tcr-widgetfield-input">
		        <input class="tcr-width-50" id="<?php echo $this->get_field_id('count'); ?>" name="<?php echo $this->get_field_name('count'); ?>" type="text" value="<?php echo esc_attr($instance['count']); ?>" />
            </div>
        </div>

        <div class="tcr-widgetfield-wrap">
            <div class="tcr-widgetfield-label">
                <label for="<?php echo $this->get_field_id('show_thumbnail'); ?>"><?php esc_html_e('Show thumbnail', TCR_ADMIN_TEXT_DOMAIN); ?></label>
            </div>
            <div class="tcr-widgetfield-input">
		        <input id="<?php echo $this->get_field_id('show_thumbnail'); ?>" name="<?php echo $this->get_field_name('show_thumbnail'); ?>" type="checkbox" <?php checked($instance['show_thumbnail'], true); ?> value="1" />
            </div>
        </div>

        <div class="tcr-widgetfield-wrap">
            <div class="tcr-widgetfield-label">
                <label for="<?php echo $this->get_field_id('show_information'); ?>"><?php esc_html_e('Extra information', TCR_ADMIN_TEXT_DOMAIN); ?></label>
            </div>
            <div class="tcr-widgetfield-input">
		        <select id="<?php echo $this->get_field_id('show_information'); ?>" name="<?php echo $this->get_field_name('show_information'); ?>">
		            <option value="date" <?php selected($instance['show_information'], 'date'); ?>><?php esc_html_e('Date', TCR_ADMIN_TEXT_DOMAIN); ?></option>
		            <option value="description" <?php selected($instance['show_information'], 'description'); ?>><?php esc_html_e('Description', TCR_ADMIN_TEXT_DOMAIN); ?></option>
		            <option value="both" <?php selected($instance['show_information'], 'both'); ?>><?php esc_html_e('Date and description', TCR_ADMIN_TEXT_DOMAIN); ?></option>
		        </select>
                <p class="tcr-widgetfield-description"><?php esc_html_e('Other post information to display', TCR_ADMIN_TEXT_DOMAIN); ?></p>
            </div>
        </div>

        <div class="tcr-widgetfield-wrap">
            <div class="tcr-widgetfield-label">
                <label for="<?php echo $this->get_field_id('title_length'); ?>"><?php esc_html_e('Title length', TCR_ADMIN_TEXT_DOMAIN); ?></label>
            </div>
            <div class="tcr-widgetfield-input">
		        <input class="tcr-width-50" id="<?php echo $this->get_field_id('title_length'); ?>" name="<?php echo $this->get_field_name('title_length'); ?>" type="text" value="<?php echo esc_attr($instance['title_length']); ?>" />
                <p class="tcr-widgetfield-description"><?php esc_html_e('Limit the post title to this number of characters', TCR_ADMIN_TEXT_DOMAIN); ?></p>
            </div>
        </div>

        <div class="tcr-widgetfield-wrap">
            <div class="tcr-widgetfield-label">
                <label for="<?php echo $this->get_field_id('description_length'); ?>"><?php esc_html_e('Description length', TCR_ADMIN_TEXT_DOMAIN); ?></label>
            </div>
            <div class="tcr-widgetfield-input">
		        <input class="tcr-width-50" id="<?php echo $this->get_field_id('description_length'); ?>" name="<?php echo $this->get_field_name('description_length'); ?>" type="text" value="<?php echo esc_attr($instance['description_length']); ?>" />
                <p class="tcr-widgetfield-description"><?php esc_html_e('Limit the post description to this number of characters', TCR_ADMIN_TEXT_DOMAIN); ?></p>
            </div>
        </div>

        <div class="tcr-widgetfield-wrap">
            <div class="tcr-widgetfield-label">
                <label for="<?php echo $this->get_field_id('categories'); ?>"><?php esc_html_e('Categories', TCR_ADMIN_TEXT_DOMAIN); ?></label>
            </div>
            <div class="tcr-widgetfield-input">
		        <select id="<?php echo $this->get_field_id('categories'); ?>" name="<?php echo $this->get_field_name('categories'); ?>[]" multiple="multiple">
		            <?php $categories = tcr_get_all_category_options(); ?>
		            <?php foreach ($categories as $categoryId => $categoryName) : ?>
		                <option value="<?php echo $categoryId; ?>" <?php if (in_array($categoryId, $instance['categories'])) echo 'selected="selected"';  ?>><?php echo esc_html($categoryName); ?></option>
		            <?php endforeach; ?>
		        </select>
                <p class="tcr-widgetfield-description"><?php esc_html_e('Show only posts from these categories', TCR_ADMIN_TEXT_DOMAIN); ?></p>
            </div>
        </div>

        <div class="tcr-widgetfield-wrap">
            <div class="tcr-widgetfield-label">
                <label for="<?php echo $this->get_field_id('authors'); ?>"><?php esc_html_e('Authors', TCR_ADMIN_TEXT_DOMAIN); ?></label>
            </div>
            <div class="tcr-widgetfield-input">
		        <select id="<?php echo $this->get_field_id('authors'); ?>" name="<?php echo $this->get_field_name('authors'); ?>[]" multiple="multiple">
		            <?php $authors = tcr_get_all_author_options(); ?>
		            <?php foreach ($authors as $authorId => $authorName) : ?>
		                <option value="<?php echo $authorId; ?>" <?php if (in_array($authorId, $instance['authors'])) echo 'selected="selected"';  ?>><?php echo esc_html($authorName); ?></option>
		            <?php endforeach; ?>
		        </select>
                <p class="tcr-widgetfield-description"><?php esc_html_e('Show only posts from these authors', TCR_ADMIN_TEXT_DOMAIN); ?></p>
            </div>
        </div>
<?php
	}

    function flush_widget_cache()
    {
		wp_cache_delete('tcr_widget_popular', 'widgets');
	}
}