<?php

/**
 * theme.php
 *
 * Front-end only functions
 */

add_filter('body_class', 'tcr_body_classes');

/**
 * Add body classes
 *
 * @param array $classes
 * @return array
 */
function tcr_body_classes($classes)
{
    global $tcr;

    $classes[] = $tcr['layout'];

    // Set sticky header (not on devices)
    if (apply_atomic('sticky_header', $tcr['options']['general_sticky_header']) && !$tcr['browser']['is_tablet_or_mobile']) {
        $classes[] = 'sticky-header';
    }

    $classes[] = apply_atomic('header_style', $tcr['options']['general_header_style']);

    // Set absolute footer (force on for devices)
    if (apply_atomic('footer_position', $tcr['options']['footer_position']) == 'absolute' || $tcr['browser']['is_tablet_or_mobile']) {
    	$classes[] = 'footer-absolute';
    }

    if ($tcr['browser']['is_tablet']) {
        $classes[] = 'is-tablet';
    }

    if ($tcr['browser']['is_mobile']) {
        $classes[] = 'is-mobile';
    }

    if ($tcr['browser']['is_tablet_or_mobile']) {
        $classes[] = 'is-tablet-or-mobile';
    }

    return $classes;
}

add_action('tcr_body_start', 'tcr_nojs_background');

function tcr_nojs_background()
{
    global $tcr; ?>
    <?php if (isset($tcr['options']['background_backgrounds'][0]['url'])) : ?>
        <div id="nojs-background">
            <img src="<?php echo site_url($tcr['options']['background_backgrounds'][0]['url']); ?>" alt="" />
        </div>
    <?php endif; ?>
    <?php
}

add_action('tcr_body_start', 'tcr_background_video');

function tcr_background_video()
{
    global $tcr;
    if (isset($tcr['video'])) {
        echo $tcr['video']['html'];
    }
}

add_action('tcr_body_start', 'tcr_background_overlay');

function tcr_background_overlay()
{
    global $tcr, $post;

    $overlay = $tcr['options']['background_overlay'];

    if (!empty($post)) {
        $overlay = tcr_get_post_meta($post->ID, 'overlay', $overlay);
    } ?>

    <?php if ($overlay != 'none') : ?>
		<div class="background-overlay <?php echo esc_attr($overlay); ?>"></div>
    <?php endif; ?>
    <?php
}

add_action('tcr_body_end', 'tcr_cufon_now');

function tcr_cufon_now()
{
    global $tcr;
    if (isset($tcr['font']) && $tcr['font']['type'] === 'cufon') : ?>
<script type="text/javascript">
if (!!window.Cufon) {
    Cufon.now();
}
</script>
    <?php endif;
}

add_action('tcr_body_end', 'tcr_analytics');

function tcr_analytics()
{
    global $tcr;
    echo $tcr['options']['general_analytics'];
}

add_action('wp_enqueue_scripts', 'tcr_enqueue_styles');

function tcr_enqueue_styles()
{
    global $tcr;
    wp_enqueue_style(TCR_SLUG, TCR_URL . '/style.css', array(), TCR_VERSION);
    wp_enqueue_style('fancybox', TCR_URL . '/js/fancybox/jquery.fancybox-1.3.4.css', array(), '1.3.4');
    wp_enqueue_style('qtip', TCR_URL . '/js/qtip2/jquery.qtip.min.css');

    if (file_exists(TCR_CACHE_DIR . '/custom.css')) {
        wp_enqueue_style('tcr-custom-css', TCR_CACHE_URL . '/custom.css', array(), $tcr['options']['last_saved']);
    }
}

add_action('tcr_head', 'tcr_google_font');

function tcr_google_font()
{
    global $tcr;
    if (isset($tcr['font']) && $tcr['font']['type'] === 'google') {
        echo '<link href="http://fonts.googleapis.com/css?family=' . $tcr['font']['name'] . '&amp;subset=' . $tcr['font']['subset'] . '" rel="stylesheet" type="text/css" />';
    }
}
add_action('wp_enqueue_scripts', 'tcr_enqueue_scripts');

function tcr_enqueue_scripts()
{
    global $tcr;

    wp_register_script('jquery-cycle', TCR_URL . '/js/jquery.cycle.min.js', array('jquery'), '2.9999.3', true);
    wp_register_script('jquery-actual', TCR_URL . '/js/jquery.actual.min.js', array('jquery'), '1.0.6', true);
    wp_register_script('jquery-cookie', TCR_URL . '/js/jquery.cookie.js', array('jquery'), '1.0', true);
    wp_register_script('jquery-hoverIntent', TCR_URL . '/js/jquery.hoverIntent.min.js', array('jquery'), '2011.02.26', true);
    wp_register_script('superfish', TCR_URL . '/js/superfish.js', array('jquery', 'jquery-hoverIntent'), '1.4.8', true);
    wp_register_script('cufon', TCR_URL . '/js/cufon.yui.js', array(), '1.09i', true);
    wp_register_script('fullscreen', TCR_URL . '/js/jquery.fullscreen.min.js', array('jquery', 'jquery-cookie', 'jquery-easing'), '1.1', true);
    wp_register_script('jquery-smooth-scroll', TCR_URL . '/js/jquery.smooth-scroll.min.js', array('jquery'), '1.4.1', true);
    wp_register_script('jquery-tools-tabs', TCR_URL . '/js/jquery.tools.tabs.min.js', array('jquery'), '1.2.6', true);
    wp_register_script('jquery-easing', TCR_URL . '/js/jquery.easing.min.js', array('jquery'), '1.3', true);
    wp_register_script('jquery-quicksand', TCR_URL . '/js/jquery.quicksand.min.js', array('jquery', 'jquery-easing'), '1.2.2', true);
    wp_register_script('fancybox', TCR_URL . '/js/fancybox/jquery.fancybox-1.3.4.pack.js', array('jquery', 'swfobject'), '1.3.4', true);
    wp_register_script('galleria', TCR_URL . '/js/galleria/galleria-1.2.7.min.js', array('jquery'), '1.3.4', true);
    wp_register_script('serene', TCR_URL . '/js/jquery.serene.min.js', array('jquery', 'jquery-cookie', 'jquery-easing'), '1.1', true);
    wp_register_script('tcr-twitter', TCR_URL . '/js/jquery.tweet.js', array('jquery'), '1.0', true);
    wp_register_script('froogaloop', 'http://a.vimeocdn.com/js/froogaloop2.min.js', array(), false, true);
    wp_register_script('jquery-throttle', TCR_URL . '/js/jquery.ba-throttle-debounce.min.js', array('jquery'), '1.1', true);
    wp_register_script('jplayer', TCR_URL . '/js/jquery.jplayer.min.js', array('jquery'),  '2.1.0', true);
    wp_register_script('qtip', TCR_URL . '/js/qtip2/jquery.qtip.min.js', array('jquery'), false, true);
    wp_register_script('infield-labels', TCR_URL . '/js/jquery.infield.min.js', array('jquery'), false, true);

    if (isset($tcr['font']) && $tcr['font']['type'] === 'cufon') {
        wp_enqueue_script('tcr-cufon-font', TCR_URL . '/fonts/' . $tcr['font']['name'] . '.font.js', array('cufon'), TCR_VERSION, true);
    }

    // Theme scripts
    wp_enqueue_script('tcr-scripts', TCR_URL . '/js/scripts.js', array('jquery', 'superfish', 'fullscreen', 'jquery-smooth-scroll', 'jquery-tools-tabs', 'fancybox', 'jquery-cycle', 'jquery-actual', 'jquery-throttle', 'qtip', 'infield-labels', 'jquery-cookie'), TCR_VERSION, true);

    // Custom scripts
    if (file_exists(TCR_CACHE_DIR . '/custom.js')) {
        wp_enqueue_script('tcr-custom-scripts', TCR_CACHE_URL . '/custom.js', array('jquery'), $tcr['options']['last_saved'], true);
    }

    if (is_singular() && get_option('thread_comments')) {
        wp_enqueue_script('comment-reply');
    }

    wp_localize_script('tcr-scripts', 'tcrL10n', tcr_script_l10n());
}

add_action('wp_footer', 'tcr_print_footer_scripts', 9);

function tcr_print_footer_scripts()
{
    global $tcr;

    if (isset($tcr['quicksand'])) {
        wp_print_scripts('jquery-quicksand');
    }

    if (isset($tcr['galleria'])) {
        wp_print_scripts('galleria');
    }

    if (isset($tcr['serene'])) {
        wp_print_scripts('serene');
    }

    if (isset($tcr['twitter'])) {
        wp_print_scripts('tcr-twitter');
    }

    if (isset($tcr['video']) && $tcr['video']['type'] == 'vimeo') {
        wp_print_scripts('froogaloop');
    }

    if (isset($tcr['audio'])) {
        wp_print_scripts('jplayer');
    }
}

add_action('wp_footer', 'tcr_print_custom_js', 30);

function tcr_print_custom_js()
{
    global $tcr;

    if (isset($tcr['footer_js'])) : ?>
<script type="text/javascript">
//<![CDATA[
    <?php echo join("\n\n", $tcr['footer_js']) . "\n"; ?>
//]]>
</script>
    <?php endif;
}

function tcr_script_l10n()
{
    global $tcr;

    $data = array(
        'themeUrl' => TCR_URL,
        'siteUrl' => site_url(),
        'backgroundOptions' => tcr_background_options(),
        'sereneOptions' => apply_atomic('serene_options', array(
            'speed' => $tcr['options']['portfolio_speed'],
            'transition' => $tcr['options']['portfolio_transition'],
            'fitLandscape' => $tcr['options']['portfolio_fit_landscape'],
            'fitPortrait' => $tcr['options']['portfolio_fit_portrait'],
            'fitAlways' => $tcr['options']['portfolio_fit_always'],
            'positionX' => $tcr['options']['portfolio_position_x'],
            'positionY' => $tcr['options']['portfolio_position_y'],
            'easing' => $tcr['options']['portfolio_easing'],
            'controlSpeed' => $tcr['options']['portfolio_control_speed'],
            'slideshow' => $tcr['options']['portfolio_slideshow'],
            'slideshowAuto' => $tcr['options']['portfolio_slideshow_auto'],
            'slideshowSpeed' => $tcr['options']['portfolio_slideshow_speed'],
            'keyboard' => $tcr['options']['portfolio_keyboard'],
            'captionPosition' => $tcr['options']['portfolio_caption_position'],
            'captionSpeed' => $tcr['options']['portfolio_caption_speed'],
            'bullets' => $tcr['options']['portfolio_bullets'],
            'lowQuality' => $tcr['options']['portfolio_low_quality'],
        	'errorBackground' => TCR_URL . '/images/backgrounds/error.jpg'
        )),
        'hideMap' => tcr__('hide_map', __('Hide map', TCR_TEXT_DOMAIN))
    );

    if (isset($tcr['video'])) {
        $data['video'] = $tcr['video'];
    }

    if (isset($tcr['audio'])) {
        $data['audio'] = $tcr['audio'];
    }

    if (isset($tcr['font'])) {
        $data['font'] = $tcr['font'];
    }

    return array('l10n_print_after' => 'tcrL10n = ' . json_encode($data) . ';');
}

add_action('wp', 'tcr_set_features');

/**
 * Set up the theme features
 */
function tcr_set_features()
{
    tcr_detect_mobile();
    tcr_set_layout();
    tcr_set_background();
    tcr_set_video_background();
    tcr_set_audio_background();
    tcr_set_font();
}

/**
 * Detect mobile/tablet
 */
function tcr_detect_mobile()
{
    global $tcr;

    if (!class_exists('Mobile_Detect')) {
        require_once TCR_INCLUDES_DIR . '/Mobile_Detect.php';
    }

    $detect = new Mobile_Detect();
    $tcr['browser']['is_tablet'] = $detect->isTablet();
    $tcr['browser']['is_mobile'] = $detect->isMobile();
    $tcr['browser']['is_tablet_or_mobile'] = $tcr['browser']['is_tablet'] || $tcr['browser']['is_mobile'];
}

function tcr_set_background()
{
    global $tcr, $post;
    $specific = false;

    $backgrounds = $tcr['options']['background_backgrounds'];

    if (!empty($post)) {
        // Check for page specific background groups
        $backgroundGroup = tcr_get_post_meta($post->ID, 'background_group');

        if (is_numeric($backgroundGroup)) {
            if (isset($tcr['options']['background_groups'][$backgroundGroup]['backgrounds'])) {
                $backgrounds = $tcr['options']['background_groups'][$backgroundGroup]['backgrounds'];
                $specific = true;
            }
        }

        // Check for page specific backgrounds
        $pageSpecificBackgrounds = tcr_get_post_meta($post->ID, 'backgrounds');

        if (is_array($pageSpecificBackgrounds) && !empty($pageSpecificBackgrounds)) {
            $backgrounds = $pageSpecificBackgrounds;
            $specific = true;
        }
    }

    $tcr['backgrounds'] = $backgrounds;
    $tcr['backgrounds_specific'] = $specific;
}

function tcr_background_options()
{
    global $tcr, $post;
    $alwaysShowCaptions = false;

    if (!empty($post)) {
        // Check the always show captions option
        if (is_page_template('template-left-note-block.php') || is_page_template('template-intro-page.php') || is_page_template('template-no-content-style.php')) {
            $alwaysShowCaptions = (bool) tcr_get_post_meta(get_the_ID(), 'always_show_captions');
        }
    }

    $bgs = array();
    foreach ($tcr['backgrounds'] as $background) {
        $bgs[] = array(
            'image' => site_url() . $background['url'],
            'title' => $background['title'],
            'caption' => $background['caption'],
            'captionPosition' => $background['captionPosition']
        );
    }

    return apply_atomic('background_options', array(
        'backgrounds' => $bgs,
        'speed' => $tcr['options']['background_speed'],
        'transition' => $tcr['options']['background_transition'],
        'position' => $tcr['options']['background_position'],
        'fitLandscape' => $tcr['options']['background_fit_landscape'],
        'fitPortrait' => $tcr['options']['background_fit_portrait'],
        'fitAlways' => $tcr['options']['background_fit_always'],
        'positionX' => $tcr['options']['background_position_x'],
        'positionY' => $tcr['options']['background_position_y'],
        'easing' => $tcr['options']['background_easing'],
        'hideSpeed' => $tcr['options']['background_hide_speed'],
        'showSpeed' => $tcr['options']['background_show_speed'],
        'controlSpeed' => $tcr['options']['background_control_speed'],
        'save' => $tcr['backgrounds_specific'] ? false : $tcr['options']['background_save'],
        'slideshow' => $tcr['options']['background_slideshow'],
        'slideshowAuto' => $tcr['options']['background_slideshow_auto'],
        'slideshowSpeed' => $tcr['options']['background_slideshow_speed'],
        'random' => $tcr['options']['background_random'],
        'keyboard' => $tcr['options']['background_keyboard'],
        'captionPosition' => $tcr['options']['background_caption_position'],
        'alwaysShowCaptions' => $alwaysShowCaptions,
        'captionSpeed' => $tcr['options']['background_caption_speed'],
        'bullets' => $tcr['options']['background_bullets'],
        'lowQuality' => $tcr['options']['background_low_quality'],
        'breaker' => $tcr['browser']['is_tablet_or_mobile'] ? false : $tcr['options']['background_breaker'],
        'breakerOnMax' => $tcr['options']['background_breaker_on_max'],
    	'errorBackground' => TCR_URL . '/images/backgrounds/error.jpg'
    ));
}

/**
 * Comments list
 */
function tcr_comment_list($comment, $args, $depth)
{
    switch ($comment->comment_type) {
        case 'pingback':
        case 'trackback':
            tcr_ping($comment, $args, $depth);
            break;
        default:
            tcr_comment($comment, $args, $depth);
            break;
    }
}

/**
 * The number of comments, will use the translation from the
 * options panel if supplied,  this only supports languages with two
 * plural forms. For further plural support, a translation of the .pot
 * file will be required.
 *
 * @param int $count
 * @return string
 */
function tcr_comment_count()
{
    global $tcr;

    $singular = tcr__('singular_comments');
    $plural = tcr__('plural_comments');

    if ($singular !== '' && $plural !== '') {
        // There is a translation in the options panel
        return tcr_get_comment_count('comment') == 1 ? $singular : $plural;
    } else {
        return _n('%d Comment', '%d Comments', tcr_get_comment_count(), TCR_TEXT_DOMAIN);
    }
}

/**
 * The number of trackbacks, will use the translation from the
 * options panel if supplied,  this only supports languages with two
 * plural forms. For further plural support, a translation of the .pot
 * file will be required.
 *
 * @param int $count
 * @return string
 */
function tcr_trackback_count()
{
    global $tcr;

    $singular = tcr__('singular_trackbacks');
    $plural = tcr__('plural_trackbacks');

    if ($singular !== '' && $plural !== '') {
        // There is a translation in the options panel
        return tcr_get_comment_count('pings') == 1 ? $singular : $plural;
    } else {
        return _n('%d Trackback', '%d Trackbacks', tcr_get_trackback_count(), TCR_TEXT_DOMAIN);
    }
}

/**
 * Get the count of comments for the current post
 *
 * @return int
 */
function tcr_get_comment_count()
{
    global $wp_query;
    $count = 0;

    if (isset($wp_query->comments_by_type, $wp_query->comments_by_type['comment'])) {
        $count = count($wp_query->comments_by_type['comment']);
    }

    return $count;
}

/**
 * Get the count of trackbacks for the current post
 *
 * @return int
 */
function tcr_get_trackback_count()
{
    global $wp_query;
    $count = 0;

    if (isset($wp_query->comments_by_type, $wp_query->comments_by_type['pings'])) {
        $count = count($wp_query->comments_by_type['pings']);
    }

    return $count;
}

/**
 * Template for comments
 */
function tcr_comment($comment, $args, $depth)
{
    $GLOBALS['comment'] = $comment;
    $url = esc_url(get_comment_author_url());
    $avatar = get_avatar($comment, 40);
    if (strlen($url)) $avatar = '<a class="comment-avatar-link" rel="external nofollow" href="' . $url . '" title="'.esc_attr(get_comment_author()).'">' . $avatar . '</a>';
    ?>
    <li <?php comment_class(); ?> id="li-comment-<?php comment_ID(); ?>">
        <div id="comment-<?php comment_ID(); ?>" class="comment">
            <div class="comment-author vcard">
                <?php edit_comment_link(__('Edit', TCR_ADMIN_TEXT_DOMAIN), '<span class="edit-link">', '</span>');
                echo $avatar; ?>
                <h2><?php printf(tcr__('author_says', __('%s <span class="says">says:</span>', TCR_TEXT_DOMAIN)), get_comment_author_link()); ?></h2>
            </div><!-- .comment-author .vcard -->

            <div class="comment-meta">
                <a href="<?php echo esc_url(get_comment_link()); ?>"><?php printf(tcr__('date_at_time', __('%1$s at %2$s', TCR_TEXT_DOMAIN)), get_comment_date(), get_comment_time()); ?></a>
            </div><!-- .comment-meta -->

            <?php if ( $comment->comment_approved == '0' ) : ?>
                <div class="comment-awaiting-moderation"><?php tcr_esc_html_e('comment_moderation', __('Your comment is awaiting moderation.', TCR_TEXT_DOMAIN)); ?></div>
            <?php endif; ?>

            <div class="comment-content"><?php comment_text(); ?></div>

            <div class="reply">
                <?php comment_reply_link(array_merge($args, array('reply_text' => tcr__('reply', __('Reply <span>&darr;</span>', TCR_TEXT_DOMAIN)), 'depth' => $depth, 'max_depth' => $args['max_depth']))); ?>
            </div><!-- .reply -->
        </div><!-- #comment-## -->
    <?php
}

/**
 * Template for pings
 */
function tcr_ping($comment, $args, $depth)
{
    $GLOBALS['comment'] = $comment; ?>
    <li class="post pingback">
        <p><?php tcr_esc_html_e('pingback', __('Pingback:', TCR_TEXT_DOMAIN)); ?> <?php comment_author_link(); ?><?php edit_comment_link(__('Edit', TCR_ADMIN_TEXT_DOMAIN), '<span class="edit-link">', '</span>'); ?></p>
    <?php
}

/**
 * Returns the line that says when the entry was posted
 */
function tcr_posted_on()
{
    global $post;
    $author = get_userdata($post->post_author);

    return sprintf(tcr__('date_by_author', __('%1$s by <a href="%2$s" title="%3$s">%4$s</a>', TCR_TEXT_DOMAIN)),
        get_the_time(get_option('date_format'), $post),
        esc_url(get_author_posts_url($author->ID)),
        esc_attr(sprintf(tcr__('view_all_posts_by', __('View all posts by %s', TCR_TEXT_DOMAIN)), $author->display_name)),
        $author->display_name
    );
}

add_action('tcr_header', 'tcr_logo');

function tcr_logo()
{
    global $tcr;
    ?><div class="logo">
        <a href="<?php echo apply_atomic('logo_url', home_url()); ?>">
            <?php if (strlen($tcr['options']['general_logo'])) : ?>
                <?php if (preg_match('#http(s)?://#', $tcr['options']['general_logo'])) : ?>
                    <img src="<?php echo esc_url($tcr['options']['general_logo']); ?>" title="" alt="Logo" />
                <?php else : ?>
                    <img src="<?php echo esc_url(site_url($tcr['options']['general_logo'])); ?>" title="" alt="Logo" />
                <?php endif; ?>
            <?php else : ?>
                <img src="<?php echo TCR_URL; ?>/images/logo.png" title="" alt="Logo" />
            <?php endif; ?>
        </a>
    </div><!-- .logo --><?php
}

add_filter('tcr_logo_url', 'tcr_logo_url');

function tcr_logo_url($url)
{
    global $tcr;

    return ($tcr['options']['general_logo_url']) ? $tcr['options']['general_logo_url'] : $url;
}

add_action('tcr_header', 'tcr_primary_menu');

function tcr_primary_menu()
{
    global $tcr;
    ?><div id="nav-wrap">
        <div id="nav-wrap-inner">
            <?php
                if (has_nav_menu('primary-menu')) {
                    echo wp_nav_menu(array(
                        'theme_location' => 'primary-menu',
                        'menu_class' => 'primary-menu',
                        'echo' => false
                    ));
                } else {
                    echo '<ul class="primary-menu">' . wp_list_pages(array(
                        'title_li' => '',
                        'echo' => false
                    )) . '</ul>';
                }
            ?>

            <?php if ($tcr['options']['general_search']) : ?>
                <div class="search-pop-up">
                    <a id="search-button" href="#" title="Search" class="search-a"></a>
                    <div class="search-container">
                        <form action="<?php echo esc_url(home_url('/')); ?>" method="get">
                            <div class="search-input-wrap">
                                <input id="search-input" value="" class="search-input" name="s" />
                            </div>
                            <div class="search-button-wrap">
                                <input name="go" type="submit" value="Go" />
                            </div>
                        </form>
                    </div>
                </div><!-- .search-pop-up -->
            <?php endif; ?>

        </div><!-- #nav-wrap-inner -->
    </div><!-- #nav-wrap --><?php
}

add_action('tcr_footer', 'tcr_display_footer');

function tcr_display_footer()
{
    global $tcr;
    $widgets = get_footer_widgets();
    if (strlen($tcr['options']['footer_left_content'])) :
    ?><div class="foot-left-col">
        <?php echo $tcr['options']['footer_left_content']; ?>
    </div><!-- .foot-left-col -->
    <?php endif; ?>
    <div class="foot-right-col">

        <?php if (count($widgets) || $tcr['options']['footer_top_link']) : ?>
            <ul class="footer-menu">
                <?php foreach ($widgets as $widget) : ?>
                <li><a id="footer-widget-area-<?php echo $widget['id']; ?>" class="footer-pop-out-trigger"><span class="up-ico"><?php echo $widget['title']; ?></span></a></li>
                <?php endforeach; ?>
                <?php if ($tcr['options']['footer_top_link']) : ?>
                <li class="top"><a href="#top" class="scroll-top" title="<?php tcr_esc_attr_e('scroll_to_top', __('Scroll to top', TCR_TEXT_DOMAIN)); ?>"><span class="up-ico"><?php tcr_esc_html_e('top', __('Top', TCR_TEXT_DOMAIN)); ?></span></a></li>
                <?php endif; ?>
            </ul>
        <?php endif; ?>
        <?php if (isset($tcr['sidebars']['footer'])) : ?>
            <?php dynamic_sidebar($tcr['sidebars']['footer']); ?>
        <?php endif;?>

    </div><!-- .foot-right-col -->
    <div id="fs-controls"><span class="label"><?php tcr_esc_html_e('background', __('Background', TCR_TEXT_DOMAIN)); ?></span></div>
    <?php if (isset($tcr['video'])) : ?>
    	<div id="video-controls"><span class="label"><?php tcr_esc_html_e('video', __('Video', TCR_TEXT_DOMAIN)); ?></span></div>
    <?php endif; ?>
    <?php if (isset($tcr['audio'])) : ?>
    	<div id="audio-controls"><span class="label"><?php tcr_esc_html_e('audio', __('Audio', TCR_TEXT_DOMAIN)); ?></span></div>
    	<div id="jplayer-container"></div>
    <?php endif; ?>
    <?php
}

function get_footer_widgets()
{
    global $tcr, $wp_registered_sidebars, $wp_registered_widgets;
    $widgets = array();

    if (isset($tcr['sidebars']['footer'])) {
        $index = $tcr['sidebars']['footer'];
        $sidebars_widgets = wp_get_sidebars_widgets();

        if (!empty($sidebars_widgets) && isset($wp_registered_sidebars[$index]) && array_key_exists($index, $sidebars_widgets) && is_array($sidebars_widgets[$index]) && !empty($sidebars_widgets[$index])) {
            foreach ($sidebars_widgets[$index] as $id) {
                if (isset($wp_registered_widgets[$id])) {
                    $widget = $wp_registered_widgets[$id];
                    if (isset($widget['callback'][0]) && $widget['callback'][0] instanceof WP_Widget) {
                        $widgetObject = $widget['callback'][0];
                        $settings = get_option($widgetObject->option_name);

                        if (is_array($settings) && isset($widget['params'][0]['number'], $settings[$widget['params'][0]['number']], $settings[$widget['params'][0]['number']]['title'])) {
                            $widgets[] = array(
                                'id' => $widget['id'],
                                'title' => $settings[$widget['params'][0]['number']]['title']
                            );
                        }
                    }
                }
            }
        }
    }

	return $widgets;
}

function tcr_404()
{
    if (is_search()) : ?>
        <p><?php tcr_esc_html_e('no_posts_found', __('Sorry, no posts matched your criteria. Try another search?', TCR_TEXT_DOMAIN)); ?></p>
    <?php else : ?>
        <p><?php tcr_esc_html_e('page_not_found', __('Sorry, but the page you requested could not be found. Perhaps searching will help.', TCR_TEXT_DOMAIN)); ?></p>
    <?php endif;
    get_search_form();
}

add_action('tcr_after_header', 'tcr_intro');

function tcr_intro()
{
    global $post, $tcr;
    $title = '';
    $intro = '';
    $subtitle = '';

    /*if (is_front_page()) {
        return;
    }*/

    if (is_home()) {
        $title = $tcr['options']['blog_title'];
        $subtitle = $tcr['options']['blog_subtitle'];
    }

    if (is_singular()) {
        $title = get_the_title($post->ID);

        if ('post' == get_post_type($post)) {
            $subtitle = tcr_posted_on();
        }

        $metaTitle = tcr_get_post_meta($post->ID, 'intro_title');
        $metaSubtitle = tcr_get_post_meta($post->ID, 'intro_subtitle');

        if (strlen($metaTitle)) {
            $title = $metaTitle;
        }

        if (strlen($metaSubtitle)) {
            $subtitle = $metaSubtitle;
        }
    }

    if (is_search()) {
        $title = tcr__('search', __('Search', TCR_TEXT_DOMAIN));
        $subtitle = sprintf(tcr__('search_results_for', __('Search results for &#8220;%s&#8221;', TCR_TEXT_DOMAIN)), '<span>' . get_search_query() . '</span>');
    }

    if (is_404()) {
        $title = tcr__('uh_oh', __('Uh oh...', TCR_TEXT_DOMAIN));
        $subtitle = tcr__('404', __('404 Not Found', TCR_TEXT_DOMAIN));
    }

    if (is_archive()) {
        $title = tcr__('archives', __('Archives', TCR_TEXT_DOMAIN));

        if (is_category()) {
            $subtitle = sprintf(tcr__('category_archive_for', __('Category Archive for %1$s', TCR_TEXT_DOMAIN)), '&lsquo;' . single_cat_title('', false) . '&rsquo;');
        } elseif (is_tag()) {
            $subtitle = sprintf(tcr__('posts_tagged', __('Posts Tagged %1$s', TCR_TEXT_DOMAIN)), '&lsquo;' . single_tag_title('', false) . '&rsquo;');
        } elseif (is_day()) {
            $subtitle = sprintf(tcr__('daily_archive_for', __('Daily Archive for %1$s', TCR_TEXT_DOMAIN)), get_the_time('F jS, Y'));
        } elseif (is_month()) {
            $subtitle = sprintf(tcr__('monthly_archive_for', __('Monthly Archive for %1$s', TCR_TEXT_DOMAIN)), get_the_time('F, Y'));
        } elseif (is_year()) {
            $subtitle = sprintf(tcr__('yearly_archive_for', __('Yearly Archive for %1$s', TCR_TEXT_DOMAIN)), get_the_time('Y'));
        } elseif (is_author()) {
            global $author;
            $authorData = get_userdata(intval($author));
            $subtitle = sprintf(tcr__('author_archive_for', __('Author Archive for %1$s', TCR_TEXT_DOMAIN)), $authorData->nickname);
        } elseif (is_tax()) {
            $term = get_term_by('slug', get_query_var('term'), get_query_var('taxonomy'));
            $subtitle = sprintf(tcr__('archives_for', __('Archives for %1$s', TCR_TEXT_DOMAIN)), '&lsquo;' . $term->name . '&rsquo;');
        }
    }

    if (strlen($title)) {
        $intro .= '<h1 class="intro-title">' . apply_atomic('intro_title', $title) . '</h1>';
    }

    if (strlen($subtitle)) {
        $intro .= '<h2 class="intro-subtitle">' . apply_atomic('intro_subtitle', $subtitle) . '</h2>';
    }

    if (strlen($intro)) {
        if (is_singular()) {
            $style = tcr_get_post_meta($post->ID, 'intro_style', $tcr['options']['general_intro_style']);
        } else {
            $style = $tcr['options']['general_intro_style'];
        }

        $style = apply_atomic('intro_style', $style);

        if ($style != 'disabled') {
            $xhtml = '<div class="intro-wrap"><div class="intro-style ' . $style . '">';
            $xhtml .= $intro;
            $xhtml .= '</div></div>';
            echo apply_atomic_shortcode('intro', $xhtml);
        }
    }
}

function tcr_content_style_class($echo = true)
{
    global $post, $tcr;

    if (is_singular()) {
        $style = tcr_get_post_meta($post->ID, 'content_style', $tcr['options']['general_content_style']);
    } else {
        $style = $tcr['options']['general_content_style'];
    }

    $style = apply_atomic('content_style', $style);

    if ($echo)
        echo $style;

    return $style;
}

function tcr_set_layout()
{
    global $tcr, $post;
    $layout = null;

    if (is_page()) {
        $layout = tcr_get_post_meta($post->ID, 'layout', $tcr['options']['general_page_layout']);
    } elseif (is_singular()) {
        if (get_post_type() == 'portfolio') {
            $layout = tcr_get_post_meta($post->ID, 'layout', $tcr['options']['portfolio_single_layout']);
        } else {
            $layout = tcr_get_post_meta($post->ID, 'layout', $tcr['options']['blog_single_layout']);
        }
    } elseif (tcr_is_blog()) {
        $layout = tcr_get_post_meta($post->ID, 'layout', $tcr['options']['blog_layout']);
    }

    if (!$layout) {
        $layout = $tcr['options']['general_layout'];
    }

    global $content_width;
    if ($layout == 'full-width') {
        $content_width = 900;
    } else {
        $content_width = 622;
    }

    $tcr['layout'] = apply_atomic('layout', $layout);
}

function tcr_set_video_background()
{
    global $tcr, $post;

    // If we have page specific background images, don't set a video
    if ($tcr['backgrounds_specific']) {
        return;
    }

    $video['url'] = $tcr['options']['background_video'];
    $video['width'] = $tcr['options']['background_video_width'];
    $video['height'] = $tcr['options']['background_video_height'];
    $video['complete'] = $tcr['options']['background_video_complete'];
    $video['redirect'] = $tcr['options']['background_video_redirect'];
    $video['autostart'] = $tcr['options']['background_video_autostart'];
    $video['start'] = $tcr['options']['background_video_start'];

    if (!empty($post)) {
        // If we have a post specific background video, use do not inherit these settings
        if (tcr_get_post_meta($post->ID, 'background_video')) {
            $video['url'] = tcr_get_post_meta($post->ID, 'background_video');
            $video['width'] = tcr_get_post_meta($post->ID, 'background_video_width');
            $video['height'] = tcr_get_post_meta($post->ID, 'background_video_height');
            $video['start'] = tcr_get_post_meta($post->ID, 'background_video_start');
        }

        $video['complete'] = tcr_get_post_meta($post->ID, 'background_video_complete', $video['complete']);
        $video['redirect'] = tcr_get_post_meta($post->ID, 'background_video_redirect', $video['redirect']);
        $video['autostart'] = tcr_get_post_meta($post->ID, 'background_video_autostart', $video['autostart']);
    }

    $video = apply_atomic('video', $video);

    if ($video['url']) {
        // Check if we're using a mobile/tablet if so, we can't use video background
        if ($tcr['browser']['is_tablet_or_mobile']) {
            return;
        }

        if ((stripos($video['url'], 'vimeo.com') !== false)) {
             // Vimeo URL
             sscanf(parse_url($video['url'], PHP_URL_PATH), '/%d', $videoId);

             if ($videoId) {
                 $output = '<div id="video-background-wrap">';
                 $output .= '<iframe id="video-background" src="http://player.vimeo.com/video/' . $videoId . '?title=0&amp;byline=0&amp;portrait=0&amp;controls=0&amp;api=1&amp;player_id=video-background" width="1920" height="1080" frameborder="0" webkitAllowFullScreen mozallowfullscreen allowFullScreen></iframe>';
                 $output .= '</div>';

                 $tcr['video'] = apply_atomic('video_vimeo', array(
                     'type' => 'vimeo',
                     'url' => $video['url'],
                     'id' => $videoId,
                     'html' => $output,
                     'autostart' => isset($video['autostart']) ? (bool) $video['autostart'] : true,
                     'complete' => isset($video['complete']) ? $video['complete'] : 'restart',
                     'width' => isset($video['width']) ? $video['width'] : '',
                     'height' => isset($video['height']) ? $video['height'] : '',
                     'start' => isset($video['start']) ? absint($video['start']) : 0,
                     'redirect' => isset($video['redirect']) ? $video['redirect'] : ''
                 ));
             }
         } else {
             if ((stripos($video['url'], 'youtu.be') !== false)) {
                 // youtu.be URL
                 $videoId = substr(parse_url($video['url'], PHP_URL_PATH), 1);
             } else {
                 // Standard YouTube URL
                 parse_str(parse_url($video['url'], PHP_URL_QUERY), $videoUrlParts);
                 $videoId = isset($videoUrlParts['v']) ? $videoUrlParts['v'] : '';
             }

             if ($videoId) {
                 $output = '<div id="video-background-wrap">';
                 $output .= '<div id="video-background"></div>';
                 $output .= '</div>';

                 $tcr['video'] = apply_atomic('video_youtube', array(
                     'type' => 'youtube',
                     'url' => $video['url'],
                     'id' => $videoId,
                     'html' => $output,
                     'autostart' => isset($video['autostart']) ? (bool) $video['autostart'] : true,
                     'complete' => isset($video['complete']) ? $video['complete'] : 'restart',
                     'width' => isset($video['width']) ? $video['width'] : '',
                     'height' => isset($video['height']) ? $video['height'] : '',
                     'start' => isset($video['start']) ? absint($video['start']) : 0,
                     'redirect' => isset($video['redirect']) ? $video['redirect'] : ''
                 ));
             }
         }
    }
}

function tcr_set_audio_background()
{
    global $tcr, $post;

    $audio = $tcr['options']['background_audio'];
    $random = $tcr['options']['background_audio_random'];
    $autostart = $tcr['options']['background_audio_autostart'];
    $complete = $tcr['options']['background_audio_complete'];
    $specific = false;

    if (!empty($post)) {
        $metaAudio = tcr_get_post_meta($post->ID, 'audio');
        if (!empty($metaAudio)) {
            $audio = $metaAudio;
            $specific = true;
        }
        $random = tcr_get_post_meta($post->ID, 'audio_random', $random);
        $autostart = tcr_get_post_meta($post->ID, 'audio_autostart', $autostart);
        $complete = tcr_get_post_meta($post->ID, 'audio_complete', $complete);
    }

    if (is_array($audio) && count($audio)) {
        $supplied = array();

        foreach (array('m4a', 'mp3', 'oga') as $format) {
            if ($audio[0][$format]) {
                $supplied[] = $format;
            }
        }

        if ($random) {
            shuffle($audio);
        }

        $tcr['audio'] = apply_atomic('audio', array(
            'supplied' => join(',', $supplied),
            'autostart' => (bool) $autostart,
            'files' => $audio,
            'complete' => $complete,
            'specific' => $specific,
            'random' => (bool) $random
        ));
    }
}

function tcr_set_font()
{
    global $tcr;

    if ($tcr['options']['general_font'] != '' &&
        ($tcr['options']['general_font_selector'] != ''
        || $tcr['options']['general_font_fullscreen']
        || $tcr['options']['general_font_portfolio']
        || $tcr['options']['general_font_serene']
        || $tcr['options']['general_font_fancybox'])) {
        $fonts = tcr_get_fonts();

        if (array_key_exists($tcr['options']['general_font'], $fonts)) {
            $tcr['font'] = $fonts[$tcr['options']['general_font']];
            $tcr['font']['selector'] = $tcr['options']['general_font_selector'];
            $tcr['font']['fullscreen'] = $tcr['options']['general_font_fullscreen'];
            $tcr['font']['portfolio'] = $tcr['options']['general_font_portfolio'];
            $tcr['font']['serene'] = $tcr['options']['general_font_serene'];
            $tcr['font']['fancybox'] = $tcr['options']['general_font_fancybox'];

            $tcr['font'] = apply_atomic('font', $tcr['font']);
        }
    }
}

add_filter('excerpt_more', 'tcr_excerpt_more');

function tcr_excerpt_more($more)
{
    return '...';
}

add_filter('the_content_more_link', 'tcr_content_more_link');

function tcr_content_more_link($link)
{
    return '<div class="read-more-link">' . $link . '</div>';
}

function tcr_content_nav($navId)
{
    global $wp_query;

    if (!function_exists('wp_pagenavi')) {
    	if ($wp_query->max_num_pages > 1) : ?>
    		<div class="content-nav clearfix" id="<?php echo $navId; ?>">
    			<div class="nav-previous"><?php next_posts_link(tcr__('older_posts', __('<span class="meta-nav">&larr;</span> Older posts', TCR_TEXT_DOMAIN))); ?></div>
    			<div class="nav-next"><?php previous_posts_link(tcr__('newer_posts', __('Newer posts <span class="meta-nav">&rarr;</span>', TCR_TEXT_DOMAIN))); ?></div>
    		</div><!-- .content-nav -->
    	<?php endif;
    } else {
        wp_pagenavi();
    }
}

function tcr_link_pages()
{
    if (!function_exists('wp_pagenavi')) {
        wp_link_pages(array('before' => '<div class="page-link"><span>' . tcr__('pages', __('Pages', TCR_TEXT_DOMAIN)) . '</span>', 'after' => '</div>'));
    } else {
        wp_pagenavi(array('type' => 'multipart'));
    }
}

/**
 * Are we on a blog page?
 *
 * @return boolean
 */
function tcr_is_blog()
{
    global $post;
    return (((is_archive()) || (is_author()) || (is_category()) || (is_home()) || (is_single()) || (is_tag())) && ('post' == get_post_type($post)));
}

/**
 * Get the HTML for the blog featured image
 *
 * @param string $type 'left' or 'right' or other type
 * @param int|string $layout Current page layout or the image width
 * @param int $height
 * @return string
 */
function tcr_blog_featured_image($type = '', $layout = '', $height = null)
{
    if (!has_post_thumbnail()) {
        return;
    }

    global $tcr;

    // Width 0 or '' should use the current page layout width
    if ($layout == '' || $layout == 0) {
        $layout = $tcr['layout'];
    }

    if (is_numeric($layout)) {
        $width = $layout;
    } else if ($layout == 'full-width') {
        $width = 900;
    } else {
        $width = 622;
    }

    if ($type == 'left' || $type == 'right') {
        if (!is_numeric($layout)) {
            if (is_single()) {
                $width = $tcr['options']['blog_single_featured_float_width'];
            } else {
                $width = $tcr['options']['blog_featured_float_width'];
            }
        }
        if (!$height) {
            if (is_single()) {
                $height = $tcr['options']['blog_single_featured_float_height'];
            } else {
                $height = $tcr['options']['blog_featured_float_height'];
            }
        }
    }

    if (!$height) {
        $imageData = wp_get_attachment_image_src(get_post_thumbnail_id(), 'full');
        if ($tcr['options']['blog_scale_featured_height'] && !empty($imageData[1])) {
            $height = round($width * ($imageData[2]/$imageData[1]), 0);
        } else {
            $height = $tcr['options']['blog_fixed_featured_height'];
        }
    }

    $src = tcr_get_image_src(get_post_thumbnail_id(), array('width' => $width, 'height' => $height));

    $style = ($type == 'left' || $type == 'right') ? 'style="width:' . $width . '; height:' . $height . ';"' : '';

    $output = '<div class="entry-image" ' . $style . '>';

    if (is_single()) {
        $output .= tcr_get_image_tag($src, array(
            'width' => $width,
            'height' => $height,
            'alt' => get_the_title()
        ));
    } else {
        $output .= '<a href="' . get_permalink() . '">';
        $output .= tcr_get_image_tag($src, array(
            'width' => $width,
            'height' => $height,
            'alt' => get_the_title()
        ));
        $output .= '</a>';
    }

    $output .= '</div>'; // .entry-image

    return $output;
}

/**
 * Get the HTML for the portfolio featured image
 *
 * @param int|string $layout Current page layout or the image width
 * @param int $height
 * @return string
 */
function tcr_portfolio_featured_image($layout = '', $height = null)
{
    if (!has_post_thumbnail()) {
        return;
    }

    global $tcr;

    if ($layout == 'full-width') {
        $width = 900;
    } elseif (is_numeric($layout) && $layout != 0) {
        $width = $layout;
    } else {
        $width = 622;
    }

    if (!$height) {
        $imageData = wp_get_attachment_image_src(get_post_thumbnail_id(), 'full');
        if ($tcr['options']['portfolio_scale_featured_height'] && !empty($imageData[1])) {
            $height = round($width * ($imageData[2]/$imageData[1]), 0);
        } else {
            $height = $tcr['options']['portfolio_fixed_featured_height'];
        }
    }

    $src = tcr_get_image_src(get_post_thumbnail_id(), array('width' => $width, 'height' => $height));

    $output = '<div class="entry-image">';

    if (is_single()) {
        $output .= tcr_get_image_tag($src, array(
            'width' => $width,
            'height' => $height,
            'alt' => get_the_title()
        ));
    } else {
        $output .= '<a href="' . get_permalink() . '">';
        $output .= tcr_get_image_tag($src, array(
            'width' => $width,
            'height' => $height,
            'alt' => get_the_title()
        ));
        $output .= '</a>';
    }

    $output .= '</div>'; // .entry-image

    return $output;
}

function tcr_formatter($content) {
    $new_content = '';
    $pattern_full = '{(\[raw\].*?\[/raw\])}is';
    $pattern_contents = '{\[raw\](.*?)\[/raw\]}is';
    $pieces = preg_split($pattern_full, $content, -1, PREG_SPLIT_DELIM_CAPTURE);

    foreach ($pieces as $piece) {
        if (preg_match($pattern_contents, $piece, $matches)) {
            $new_content .= $matches[1];
        } else {
            $new_content .= wptexturize(wpautop($piece));
        }
    }

    return $new_content;
}

remove_filter('the_content', 'wpautop');
remove_filter('the_content', 'wptexturize');

add_filter('the_content', 'tcr_formatter', 99);

function tcr_add_footer_js($js)
{
    global $tcr;

    if (!isset($tcr['footer_js'])) {
        $tcr['footer_js'] = array();
    }

    $tcr['footer_js'][] = $js;
}

function tcr_get_paged_var()
{
    global $wp_version;
    $paged = 1;

    if (is_front_page() && version_compare($wp_version, '3.1', '>=')) {
        if (get_query_var('paged')) {
            $paged = get_query_var('paged');
        } elseif (get_query_var('page')) {
            $paged = get_query_var('page');
        }
    } else {
        if (get_query_var('paged')) {
            $paged = get_query_var('paged');
        }
    }

    return (int) $paged;
}

// Class to hold excerpt lengths for filters
class ExcerptLength {
    private $_length;

    public function __construct($length)
    {
        $this->_length = $length;
    }

    public function getLength()
    {
        return $this->_length;
    }
}

/**
 * Custom WP-PageNavi
 *
 * @param string $before HTML before the navigation
 * @param string $after HTML after the navigation
 * @param WP_Query $query The query object
 * @param int $paged Current page number
 */
function tcr_pagenavi($before = '', $after = '', $query, $paged)
{
    global $wpdb, $wp_query;

	if (is_single())
		return;

	$pagenavi_options = array(
	    'pages_text' => tcr__('pages_text', __('Page %CURRENT_PAGE% of %TOTAL_PAGES%', TCR_TEXT_DOMAIN)),
	    'current_text' => '%PAGE_NUMBER%',
	    'page_text' => '%PAGE_NUMBER%',
	    'first_text' => tcr__('first_text', __('&laquo; First', TCR_TEXT_DOMAIN)),
	    'last_text' => tcr__('last_text', __('Last &raquo;', TCR_TEXT_DOMAIN)),
	    'next_text' => tcr__('next_text', __('&raquo;', TCR_TEXT_DOMAIN)),
	    'prev_text' => tcr__('prev_text', __('&laquo;', TCR_TEXT_DOMAIN)),
	    'dotright_text' => tcr__('dotright_text', __('...', TCR_TEXT_DOMAIN)),
	    'dotleft_text' => tcr__('dotleft_text', __('...', TCR_TEXT_DOMAIN)),
	    'style' => 1,
	    'num_pages' => 4,
	    'always_show' => 0,
	    'num_larger_page_numbers' => 3,
	    'larger_page_numbers_multiple' => 10,
	    'use_pagenavi_css' => 0,
	);

	$request = $query->request;
	$posts_per_page = intval(get_query_var('posts_per_page'));

	$numposts = $query->found_posts;
	$max_page = intval($query->max_num_pages);

	if (empty($paged) || $paged == 0)
		$paged = 1;

	$pages_to_show = intval($pagenavi_options['num_pages']);
	$larger_page_to_show = intval($pagenavi_options['num_larger_page_numbers']);
	$larger_page_multiple = intval($pagenavi_options['larger_page_numbers_multiple']);
	$pages_to_show_minus_1 = $pages_to_show - 1;
	$half_page_start = floor($pages_to_show_minus_1/2);
	$half_page_end = ceil($pages_to_show_minus_1/2);
	$start_page = $paged - $half_page_start;

	if ($start_page <= 0)
		$start_page = 1;

	$end_page = $paged + $half_page_end;
	if (($end_page - $start_page) != $pages_to_show_minus_1) {
		$end_page = $start_page + $pages_to_show_minus_1;
	}

	if ($end_page > $max_page) {
		$start_page = $max_page - $pages_to_show_minus_1;
		$end_page = $max_page;
	}

	if ($start_page <= 0)
		$start_page = 1;

	$larger_pages_array = array();
	if ( $larger_page_multiple )
		for ( $i = $larger_page_multiple; $i <= $max_page; $i += $larger_page_multiple )
			$larger_pages_array[] = $i;

	if ($max_page > 1 || intval($pagenavi_options['always_show'])) {
		$pages_text = str_replace("%CURRENT_PAGE%", number_format_i18n($paged), $pagenavi_options['pages_text']);
		$pages_text = str_replace("%TOTAL_PAGES%", number_format_i18n($max_page), $pages_text);
		echo $before.'<div class="wp-pagenavi">'."\n";
		switch(intval($pagenavi_options['style'])) {
			// Normal
			case 1:
				if (!empty($pages_text)) {
					echo '<span class="pages">'.$pages_text.'</span>';
				}
				if ($start_page >= 2 && $pages_to_show < $max_page) {
					$first_page_text = str_replace("%TOTAL_PAGES%", number_format_i18n($max_page), $pagenavi_options['first_text']);
					echo '<a href="'.esc_url(get_pagenum_link()).'" class="first" title="'.$first_page_text.'">'.$first_page_text.'</a>';
					if (!empty($pagenavi_options['dotleft_text'])) {
						echo '<span class="extend">'.$pagenavi_options['dotleft_text'].'</span>';
					}
				}
				$larger_page_start = 0;
				foreach($larger_pages_array as $larger_page) {
					if ($larger_page < $start_page && $larger_page_start < $larger_page_to_show) {
						$page_text = str_replace("%PAGE_NUMBER%", number_format_i18n($larger_page), $pagenavi_options['page_text']);
						echo '<a href="'.esc_url(get_pagenum_link($larger_page)).'" class="page" title="'.$page_text.'">'.$page_text.'</a>';
						$larger_page_start++;
					}
				}
				previous_posts_link($pagenavi_options['prev_text']);
				for($i = $start_page; $i  <= $end_page; $i++) {
					if ($i == $paged) {
						$current_page_text = str_replace("%PAGE_NUMBER%", number_format_i18n($i), $pagenavi_options['current_text']);
						echo '<span class="current">'.$current_page_text.'</span>';
					} else {
						$page_text = str_replace("%PAGE_NUMBER%", number_format_i18n($i), $pagenavi_options['page_text']);
						echo '<a href="'.esc_url(get_pagenum_link($i)).'" class="page" title="'.$page_text.'">'.$page_text.'</a>';
					}
				}
				next_posts_link($pagenavi_options['next_text'], $max_page);
				$larger_page_end = 0;
				foreach($larger_pages_array as $larger_page) {
					if ($larger_page > $end_page && $larger_page_end < $larger_page_to_show) {
						$page_text = str_replace("%PAGE_NUMBER%", number_format_i18n($larger_page), $pagenavi_options['page_text']);
						echo '<a href="'.esc_url(get_pagenum_link($larger_page)).'" class="page" title="'.$page_text.'">'.$page_text.'</a>';
						$larger_page_end++;
					}
				}
				if ($end_page < $max_page) {
					if (!empty($pagenavi_options['dotright_text'])) {
						echo '<span class="extend">'.$pagenavi_options['dotright_text'].'</span>';
					}
					$last_page_text = str_replace("%TOTAL_PAGES%", number_format_i18n($max_page), $pagenavi_options['last_text']);
					echo '<a href="'.esc_url(get_pagenum_link($max_page)).'" class="last" title="'.$last_page_text.'">'.$last_page_text.'</a>';
				}
				break;
			// Dropdown
			case 2;
				echo '<form action="'.htmlspecialchars($_SERVER['PHP_SELF']).'" method="get">'."\n";
				echo '<select size="1" onchange="document.location.href = this.options[this.selectedIndex].value;">'."\n";
				for($i = 1; $i  <= $max_page; $i++) {
					$page_num = $i;
					if ($page_num == 1) {
						$page_num = 0;
					}
					if ($i == $paged) {
						$current_page_text = str_replace("%PAGE_NUMBER%", number_format_i18n($i), $pagenavi_options['current_text']);
						echo '<option value="'.esc_url(get_pagenum_link($page_num)).'" selected="selected" class="current">'.$current_page_text."</option>\n";
					} else {
						$page_text = str_replace("%PAGE_NUMBER%", number_format_i18n($i), $pagenavi_options['page_text']);
						echo '<option value="'.esc_url(get_pagenum_link($page_num)).'">'.$page_text."</option>\n";
					}
				}
				echo "</select>\n";
				echo "</form>\n";
				break;
		}
		echo '</div>'.$after."\n";
	}
}

function tcr_breadcrumbs()
{
    global $tcr;

    if ($tcr['options']['general_breadcrumbs'] && function_exists('breadcrumbs_plus')) {
        echo breadcrumbs_plus(array());
    }
}

add_action('tcr_head', 'tcr_favicon');

function tcr_favicon()
{
    global $tcr;

    if ($tcr['options']['general_favicon']) {
        if (preg_match('#http(s)?://#', $tcr['options']['general_favicon'])) {
            $href = $tcr['options']['general_favicon'];
        } else {
            $href = site_url($tcr['options']['general_favicon']);
        }

        echo '<link rel="shortcut icon" href="' . esc_url($href) . '" />';
    }
}