<?php

/**
 * framework.php
 *
 * - Theme setup
 * - Front-end and admin common functions
 */

// Prevent direct script access
if (!defined('ABSPATH')) exit;

// Define constants

defined('TCR_NAME')
    || define('TCR_NAME', 'Storm');

defined('TCR_VERSION')
    || define('TCR_VERSION', '1.1.1');

defined('TCR_SLUG')
    || define('TCR_SLUG', get_template());

defined('TCR_DIR')
    || define('TCR_DIR', get_template_directory());

defined('TCR_URL')
    || define('TCR_URL', get_template_directory_uri());

defined('TCR_INCLUDES_DIR')
    || define('TCR_INCLUDES_DIR', TCR_DIR . '/includes');

defined('TCR_CACHE_DIR')
    || define('TCR_CACHE_DIR', TCR_DIR . '/cache');

defined('TCR_CACHE_URL')
    || define('TCR_CACHE_URL', TCR_URL . '/cache');

defined('TCR_ADMIN_DIR')
    || define('TCR_ADMIN_DIR', TCR_DIR . '/admin');

defined('TCR_ADMIN_INCLUDES_DIR')
    || define('TCR_ADMIN_INCLUDES_DIR', TCR_ADMIN_DIR . '/includes');

defined('TCR_ADMIN_URL')
    || define('TCR_ADMIN_URL', TCR_URL . '/admin');

defined('TCR_OPTIONS_NAME')
    || define('TCR_OPTIONS_NAME', TCR_SLUG . '_options');

defined('TCR_TEXT_DOMAIN')
    || define('TCR_TEXT_DOMAIN', TCR_SLUG);

defined('TCR_ADMIN_TEXT_DOMAIN')
    || define('TCR_ADMIN_TEXT_DOMAIN', TCR_SLUG . '_admin');

defined('TCR_PREFIX')
    || define ('TCR_PREFIX', 'tcr');

// Get the theme options
$tcr['options'] = tcr_get_options();

// Load framework hooks
require_once TCR_INCLUDES_DIR . '/hooks.php';

// Load the theme shortcodes
require_once TCR_INCLUDES_DIR . '/shortcodes.php';

// Load the theme widgets
require_once TCR_INCLUDES_DIR . '/widgets.php';

/**
 * Get the theme options
 *
 * @return array
 */
function tcr_get_options()
{
    $savedOptions = get_option(TCR_OPTIONS_NAME);
    $savedOptions = is_array($savedOptions) ? $savedOptions : array();

    return array_merge(tcr_get_default_options(), $savedOptions);
}

add_action('after_setup_theme', 'tcr_theme_setup');

/**
 * Theme setup
 */
function tcr_theme_setup()
{
    // Set up translations
    load_theme_textdomain(TCR_TEXT_DOMAIN, TCR_DIR . '/languages');

    // Add thumbnail support
    add_theme_support('post-thumbnails');

    // Add post format support
    add_theme_support('post-formats', array('aside', 'link', 'gallery', 'status', 'quote', 'image', 'video', 'chat'));
}

add_action('init', 'tcr_register_custom_post_types');

/**
 * Register custom post types
 */
function tcr_register_custom_post_types()
{
    global $tcr;

    // Register Portfolio post type
    register_post_type('portfolio', array(
        'labels' => array(
            'name' => __('Portfolio', TCR_ADMIN_TEXT_DOMAIN),
            'singular_name' => __('Portfolio Item', TCR_ADMIN_TEXT_DOMAIN),
            'add_new' => _x('Add New', 'add portfolio item', TCR_ADMIN_TEXT_DOMAIN),
            'add_new_item' => _x('Add New Item', 'add new portfolio item', TCR_ADMIN_TEXT_DOMAIN),
            'all_items' => _x('All Items', 'all portfolio items', TCR_ADMIN_TEXT_DOMAIN),
            'edit_item' => _x('Edit Item', 'edit portfolio item', TCR_ADMIN_TEXT_DOMAIN),
            'new_item' => __('New Portfolio Item', TCR_ADMIN_TEXT_DOMAIN),
            'view_item' => _x('View Item', 'view portfolio item', TCR_ADMIN_TEXT_DOMAIN),
            'search_items' => __('Search Portfolio Items', TCR_ADMIN_TEXT_DOMAIN),
            'not_found' =>  __('No portfolio Items found.', TCR_ADMIN_TEXT_DOMAIN),
            'not_found_in_trash' => __('No portfolio items found in Trash.', TCR_ADMIN_TEXT_DOMAIN),
            'parent_item_colon' => '',
        ),
        'public' => true,
        'publicly_queryable' => true,
        'exclude_from_search' => false,
        'show_ui' => true,
        'show_in_menu' => true,
        'capability_type' => 'post',
        'hierarchical' => false,
        'supports' => array('title', 'editor', 'excerpt', 'thumbnail', 'page-attributes'),
        'has_archive' => false,
        'query_var' => false,
        'rewrite' => array('slug' => $tcr['options']['portfolio_rewrite_slug'] ? $tcr['options']['portfolio_rewrite_slug'] : 'portfolio', 'with_front' => true, 'pages' => true, 'feeds' => false),
        'can_export' => true,
        'show_in_nav_menus' => true
    ));

    if ($tcr['options']['portfolio_comments']) {
        add_post_type_support('portfolio', 'comments');
    }

    // Register Portfolio Category taxonomy
    register_taxonomy('portfolio_category', 'portfolio', array(
        'labels' => array(
            'name' => __('Portfolio Categories', TCR_ADMIN_TEXT_DOMAIN),
            'singular_name' => __('Portfolio Category', TCR_ADMIN_TEXT_DOMAIN),
            'search_items' =>  __('Search Categories', TCR_ADMIN_TEXT_DOMAIN),
            'popular_items' => __('Popular Categories', TCR_ADMIN_TEXT_DOMAIN),
            'all_items' => __('All Categories', TCR_ADMIN_TEXT_DOMAIN),
            'parent_item' => null,
            'parent_item_colon' => null,
            'edit_item' => __('Edit Portfolio Category', TCR_ADMIN_TEXT_DOMAIN),
            'update_item' => __('Update Portfolio Category', TCR_ADMIN_TEXT_DOMAIN),
            'add_new_item' => __('Add New Portfolio Category', TCR_ADMIN_TEXT_DOMAIN),
            'new_item_name' => __('New Portfolio Category Name', TCR_ADMIN_TEXT_DOMAIN),
            'menu_name' => __('Categories', TCR_ADMIN_TEXT_DOMAIN)
        ),
        'public' => false,
        'show_in_nav_menus' => false,
        'show_ui' => true,
        'show_tagcloud' => false,
        'hierarchical' => true,
        'query_var' => false,
        'rewrite' => false
    ));
}

/**
 * Get the default options for the theme
 *
 * @return array
 */
function tcr_get_default_options()
{
    $translations = tcr_get_default_translations();

    foreach ($translations as $key => &$value) {
        $value = '';
    }

    return array(
        // General
        'general_logo' => str_replace(site_url(), '', TCR_URL . '/images/logo.png'),
        'general_logo_url' => '',
        'general_logo_top' => 44,
        'general_logo_left' => 0,
        'general_favicon' => '',
        'general_sticky_header' => false,
        'general_header_style' => 'standard-header',
        'general_navigation_width' => 690,
        'general_analytics' => '',
        'general_search' => true,
        'general_layout' => 'full-width',
        'general_page_layout' => 'full-width',
        'general_intro_style' => 'transparent-intro',
        'general_content_style' => 'plain-black',
        'general_read_more_text' => __('Read More', TCR_TEXT_DOMAIN),
        'general_breadcrumbs' => true,
        'general_font' => 'comfortaa',
        'general_font_selector' => 'h1, h3.widget-title, h2.intro-subtitle',
        'general_font_fullscreen' => true,
        'general_font_portfolio' => true,
        'general_font_serene' => true,
        'general_font_fancybox' => true,
        'general_color_main' => '#ffa200',
        'general_color_heading' => '#ffffff',
        'general_color_subheading' => '#ffa200',
        'general_color_background' => '#111111',

        // Background
        'background_backgrounds' => array(),
        'background_groups' => array(),
        'background_overlay' => 'diagonal',
        'background_video' => '',
        'background_video_width' => '',
        'background_video_height' => '',
    	'background_video_start' => 0,
        'background_video_autostart' => true,
        'background_video_complete' => 'restart',
        'background_video_redirect' => '',
        'background_audio' => array(),
        'background_audio_random' => false,
    	'background_audio_autostart' => true,
        'background_audio_complete' => 'restart',
        'background_speed' => 2000,
        'background_transition' => 'fade',
        'background_position' => 'fixed',
        'background_fit_landscape' => false,
        'background_fit_portrait' => true,
        'background_fit_always' => false,
        'background_position_x' => 'center',
        'background_position_y' => 'center',
        'background_easing' => 'swing',
        'background_hide_speed' => 1000,
        'background_show_speed' => 1000,
        'background_control_speed' => 500,
        'background_save' => true,
        'background_slideshow' => true,
        'background_slideshow_auto' => true,
        'background_slideshow_speed' => 5000,
        'background_random' => false,
        'background_keyboard' => true,
        'background_caption_position' => 'right bottom',
        'background_caption_speed' => 600,
        'background_bullets' => true,
        'background_low_quality' => false,
        'background_breaker' => false,
        'background_breaker_on_max' => false,

        // Blog
        'blog_layout' => 'right-sidebar',
        'blog_single_layout' => 'right-sidebar',
        'blog_show_date_circle' => true,
        'blog_comments_layout' => 'tabs',
        'blog_title' => __('Blog', TCR_TEXT_DOMAIN),
        'blog_subtitle' => '',
        'blog_single_featured_image' => true,
        'blog_single_featured_image_type' => 'full',
        'blog_single_featured_float_width' => 250,
        'blog_single_featured_float_height' => 0,
        'blog_show_author_description' => true,
        'blog_show_single_nav' => true,
        'blog_featured_image' => true,
        'blog_featured_image_type' => 'below',
        'blog_scale_featured_height' => true,
        'blog_fixed_featured_height' => 300,
        'blog_featured_float_width' => 150,
        'blog_featured_float_height' => 0,

        // Portfolio
        'portfolio_single_layout' => 'full-width',
        'portfolio_show_single_nav' => true,
        'portfolio_featured_image' => true,
        'portfolio_scale_featured_height' => true,
        'portfolio_fixed_featured_height' => 300,
        'portfolio_video_width' => 640,
        'portfolio_video_height' => 360,
        'portfolio_comments' => false,
        'portfolio_rewrite_slug' => '',
        'portfolio_speed' => 2000,
        'portfolio_transition' => 'fade',
        'portfolio_fit_landscape' => false,
        'portfolio_fit_portrait' => true,
        'portfolio_fit_always' => false,
        'portfolio_position_x' => 'center',
        'portfolio_position_y' => 'center',
        'portfolio_easing' => 'swing',
        'portfolio_control_speed' => 500,
        'portfolio_slideshow' => true,
        'portfolio_slideshow_auto' => true,
        'portfolio_slideshow_speed' => 7000,
        'portfolio_keyboard' => true,
        'portfolio_caption_position' => 'center bottom',
        'portfolio_caption_speed' => 600,
        'portfolio_bullets' => true,
        'portfolio_low_quality' => false,

        // Footer
        'footer_left_content' => '<a href="http://demos.themecatcher.net/buy.php?product=storm-wordpress">Storm</a> by ThemeCatcher &copy; ' . date('Y') . ' ' . get_bloginfo('name'),
        'footer_top_link' => true,
    	'footer_position' => 'fixed',

        // Contact
        'contact_quform_id' => '',
        'contact_phone_number' => '',
        'contact_fax_number' => '',
        'contact_email' => '',
        'contact_address' => '',
        'contact_map' => '',

        // Advanced
        'advanced_custom_css' => '',
        'advanced_custom_js' => '',

        'last_saved' => time()
    ) + $translations;
}

/**
 * Get the list of translatable strings
 */
function tcr_get_default_translations()
{
    return array(
		'translate_location' => 'Location',
        'translate_find_us' => 'Find us',
        'translate_hide_map' => 'Hide map',
        'translate_phone' => 'Phone',
        'translate_fax' => 'Fax',
        'translate_email' => 'Email',
        'translate_contact_details' => 'Contact Details',
        'translate_facebook' => 'Facebook',
        'translate_opening_times' => 'Opening Times',
        'translate_popular_posts' => 'Popular Posts',
        'translate_recent_posts' => 'Recent Posts',
        'translate_related_posts' => 'Related Posts',
        'translate_social_links' => 'Social Pinks',
        'translate_twitter' => 'Twitter',
        'translate_loading_tweets' => 'Loading tweets...',
        'translate_follow_me' => 'Follow Me',
    	'translate_no_portfolio_items' => 'No portfolio items found.',
        'translate_permalink_to' => 'Permalink to %s',
        'translate_author_says' => '%s <span class="says">says:</span>',
        'translate_date_at_time' => '%1$s at %2$s',
        'translate_comment_moderation' => 'Your comment is awaiting moderation.',
        'translate_pingback' => 'Pingback:',
        'translate_date_by_author' => '%1$s by <a href="%2$s" title="%3$s">%4$s</a>',
        'translate_view_all_posts_by' => 'View all posts by %s',
        'translate_older_posts' => '<span class="meta-nav">&larr;</span> Older posts',
        'translate_newer_posts' => 'Newer posts <span class="meta-nav">&rarr;</span>',
        'translate_scroll_to_top' => 'Scroll to top',
        'translate_top' => 'Top',
    	'translate_no_posts_found' => 'Sorry, no posts matched your criteria. Try another search?',
        'translate_page_not_found' => 'Sorry, but the page you requested could not be found. Perhaps searching will help.',
        'translate_search' => 'Search',
        'translate_search_results_for' => 'Search results for &#8220;%s&#8221;',
        'translate_uh_oh' => 'Uh oh...',
        'translate_404' => '404 Not Found',
        'translate_archives' => 'Archives',
        'translate_category_archive_for' => 'Category Archive for %1$s',
        'translate_posts_tagged' => 'Posts Tagged %1$s',
        'translate_daily_archive_for' => 'Monthly Archive for %1$s',
    	'translate_monthly_archive_for' => 'Monthly Archive for %1$s',
        'translate_yearly_archive_for' => 'Yearly Archive for %1$s',
        'translate_author_archive_for' => 'Author Archive for %1$s',
        'translate_archives_for' => 'Archives for %1$s',
        'translate_pages' => 'Pages',
        'translate_pages_text' => 'Page %CURRENT_PAGE% of %TOTAL_PAGES%',
        'translate_first_text' => '&laquo; First',
        'translate_last_text' => 'Last &raquo;',
        'translate_next_text' => '&raquo;',
        'translate_prev_text' => '&laquo;',
    	'translate_dotright_text' => '...',
        'translate_dotleft_text' => '...',
        'translate_about_author' => 'About %s',
        'translate_singular_comments' => '%d Comment',
        'translate_plural_comments' => '%d Comments',
        'translate_singular_trackbacks' => '%d Trackback',
        'translate_plural_trackbacks' => '%d Trackbacks',
        'translate_comments' => 'Comments',
        'translate_trackbacks' => 'Trackbacks',
        'translate_reply' => 'Reply <span>&darr;</span>',
        'translate_older_comments' => '&larr; Older Comments',
    	'translate_newer_comments' => 'Newer Comments &rarr;',
        'translate_comments_closed' => 'Comments are closed.',
        'translate_name' => 'Name',
        'translate_email' => 'Email',
        'translate_website' => 'Website',
        'translate_comment' => 'Comment',
        'translate_leave_reply' => 'Leave a Reply',
        'translate_one_reply' => '<b>1</b> Reply',
        'translate_many_replies' => '<b>%</b> Replies',
        'translate_categories_colon' => 'Categories:',
        'translate_tags_colon' => 'Tags:',
        'translate_skip_intro' => 'Skip intro &raquo;',
        'translate_background' => 'Background',
        'translate_video' => 'Video',
        'translate_audio' => 'Audio',
    );
}

add_action('init', 'tcr_register_nav_menus');

function tcr_register_nav_menus() {
    register_nav_menus(array(
        'primary-menu' => __('Primary Menu', TCR_ADMIN_TEXT_DOMAIN)
    ));
}

/**
 * Get the post meta value with the given key or the default
 * if the value is empty

 * @param int $post_id
 * @param string $key
 * @param mixed $default
 * @return mixed
 */
function tcr_get_post_meta($post_id, $key, $default = '')
{
    $value = get_post_meta($post_id, '_tcr_' . $key, true);

    if ($value === '' || $value === false) {
        return $default;
    }

    return $value;
}

/**
 * Returns an HTML image element
 *
 * @param string $src Path of the image
 * @param array $atts Other attributes of the img tag
 */
function tcr_get_image_tag($src, $atts = array())
{
    if (!isset($atts['alt'])) {
        $atts['alt'] = '';
    }

    $imgAtts = '';
    foreach ($atts as $att => $value) {
        $imgAtts .= ' ' . $att . '="' . esc_attr($value) . '"';
    }

    return '<img src="' . $src . '"'.$imgAtts.' />';
}

/**
 * Get the path to the image resized to the given dimensions
 *
 * @param string|int $image Path or attachment ID
 * @param array $size An array with width and height values
 * @param int $quality The result image quality
 */
function tcr_get_image_src($image, $size = 'full', $quality = 90)
{
    if(!$image) {
        return '';
    }

    if (is_numeric($image)) {
        $image = wp_get_attachment_image_src($image, 'full');
        $image = $image[0];
    }

    if (!is_array($size)) {
        return $image;
    } else {
        if (!isset($size['width'])) $size['width'] = 0;
        if (!isset($size['height'])) $size['height'] = 0;
    }

    $params = array(
        'src' => tcr_get_image_src($image),
        'w' => $size['width'],
        'h' => $size['height'],
        'q' => $quality
    );

    return TCR_URL . '/includes/timthumb.php?' . http_build_query($params, '', '&amp;');
}

/**
 * Logging function
 *
 * Logs each argument as an entry in the error log
 */
function tcr_error_log()
{
    foreach (func_get_args() as $arg) {
        ob_start();
        var_dump($arg);
        error_log(ob_get_clean());
    }
}


/**
 * Get the list of all available fonts. Format:
 *
 * 'unique_key - preview image filename' => array(
 *     'type' => 'cufon|google',
 *     'name' => 'font file name before .font.js (Cufon)|font name in href (Google)',
 *     'family' => 'name of font family'
 *  );
 *
 * @return array
 */
function tcr_get_fonts()
{
    return apply_filters('tcr_fonts', array(
        // Cufon fonts
        'comfortaa' => array(
            'type' => 'cufon',
            'name' => 'Comfortaa_400',
            'family' => 'Comfortaa',
            'preview' => 'comfortaa.gif'
        ),
        'dejavu' => array(
            'type' => 'cufon',
            'name' => 'DejaVu_Serif_Condensed_400',
            'family' => 'DejaVu Serif Condensed',
            'preview' => 'dejavu.gif'
        ),
        'raspoutine' => array(
            'type' => 'cufon',
            'name' => 'Raspoutine_DemiBold_italic_600',
            'family' => 'Raspoutine DemiBold',
            'preview' => 'raspoutine.gif'
        ),
		'caviar_dreams' => array(
            'type' => 'cufon',
            'name' => 'Caviar_Dreams_400',
            'family' => 'Caviar Dreams',
            'preview' => 'caviar_dreams.gif'
        ),
		'cicle' => array(
            'type' => 'cufon',
            'name' => 'Cicle_300',
            'family' => 'Cicle',
            'preview' => 'cicle.gif'
        ),
		'dirty_ego' => array(
            'type' => 'cufon',
            'name' => 'Dirty_Ego_400',
            'family' => 'Dirty Ego',
            'preview' => 'dirty_ego.gif'
        ),
		'homizio' => array(
            'type' => 'cufon',
            'name' => 'Homizio_400',
            'family' => 'Homizio',
            'preview' => 'homizio.gif'
        ),
		'nilland' => array(
            'type' => 'cufon',
            'name' => 'Nilland_400',
            'family' => 'Nilland',
            'preview' => 'nilland.gif'
        ),
		'x360' => array(
            'type' => 'cufon',
            'name' => 'X360_by_Redge_400',
            'family' => 'X360 by Redge',
            'preview' => 'x360.gif'
        ),

        // Google fonts
        'oxygen' => array(
            'type' => 'google',
            'name' => 'Oxygen',
            'family' => 'Oxygen',
            'preview' => 'oxygen.gif',
            'subset' => 'latin' // also available: latin-ext
        ),
        'marmelad' => array(
            'type' => 'google',
            'name' => 'Marmelad',
            'family' => 'Marmelad',
            'preview' => 'marmelad.gif',
            'subset' => 'latin' // also available: latin-ext,cyrillic
        ),
        'dosis' => array(
            'type' => 'google',
            'name' => 'Dosis',
            'family' => 'Dosis',
            'preview' => 'dosis.gif',
            'subset' => 'latin' // also available: latin-ext
        ),
        'playball' => array(
            'type' => 'google',
            'name' => 'Playball',
            'family' => 'Playball',
            'preview' => 'playball.gif',
            'subset' => 'latin' // also available: latin-ext
        ),
        'noticia_text' => array(
            'type' => 'google',
            'name' => 'Noticia+Text',
            'family' => 'Noticia Text',
            'preview' => 'noticia_text.gif',
            'subset' => 'latin' // also available: latin-ext,vietnamese
        ),
        'pontano_sans' => array(
            'type' => 'google',
            'name' => 'Pontano+Sans',
            'family' => 'Pontano Sans',
            'preview' => 'pontano_sans.gif',
            'subset' => 'latin' // also available: latin-ext
        ),
        'brawler' => array(
            'type' => 'google',
            'name' => 'Brawler',
            'family' => 'Brawler',
            'preview' => 'brawler.gif',
            'subset' => 'latin' // no other subsets available
        ),
        'handlee' => array(
            'type' => 'google',
            'name' => 'Handlee',
            'family' => 'Handlee',
            'preview' => 'handlee.gif',
            'subset' => 'latin' // no other subsets available
        ),
        'ruluko' => array(
            'type' => 'google',
            'name' => 'Ruluko',
            'family' => 'Ruluko',
            'preview' => 'ruluko.gif',
            'subset' => 'latin' // also available: latin-ext
        ),
        'cabin_condensed' => array(
            'type' => 'google',
            'name' => 'Cabin+Condensed',
            'family' => 'Cabin Condensed',
            'preview' => 'cabin_condensed.gif',
            'subset' => 'latin' // no other subsets available
        ),
        'arimo' => array(
            'type' => 'google',
            'name' => 'Arimo',
            'family' => 'Arimo',
            'preview' => 'arimo.gif',
            'subset' => 'latin' // no other subsets available
        ),
        'tinos' => array(
            'type' => 'google',
            'name' => 'Tinos',
            'family' => 'Tinos',
            'preview' => 'tinos.gif',
            'subset' => 'latin' // no other subsets available
        ),
        'droid_serif' => array(
            'type' => 'google',
            'name' => 'Droid+Serif',
            'family' => 'Droid Serif',
            'preview' => 'droid_serif.gif',
            'subset' => 'latin' // no other subsets available
        ),
        'open_sans' => array(
            'type' => 'google',
            'name' => 'Open+Sans:400,300',
            'family' => 'Open Sans',
            'preview' => 'open_sans.gif',
            'subset' => 'latin' // also available: latin-ext,cyrillic,cyrillic-ext,greek,greek-ext,vietnamese
        ),
        'bitter' => array(
            'type' => 'google',
            'name' => 'Bitter:400,700',
            'family' => 'Bitter',
            'preview' => 'bitter.gif',
            'subset' => 'latin' // also available: latin-ext
        )
    ));
}

/**
 * Get the font options list
 *
 * @return array
 */
function tcr_get_font_options()
{
    $fonts = tcr_get_fonts();
    $options = array(
        'cufon' => array(),
        'google' => array()
    );

    foreach ($fonts as $name => $info) {
        if ($info['type'] == 'cufon') {
            $options['cufon'][$name] = $info;
        } elseif ($info['type'] == 'google') {
            $options['google'][$name] = $info;
        }
    }

    return $options;
}


/**
 * Return the translation for the given key or use $default if
 * there is no translation
 *
 * @param string $key The translation key
 * @param string $default Default translation
 */
function tcr__($key, $default = '')
{
    global $tcr;
    $key = 'translate_' . $key;
    if ($tcr['options'][$key] !== '') {
        return $tcr['options'][$key];
    }

    return $default;
}

/**
 * Echo the translation for the given key or use $default if
 * there is no translation
 *
 * @param string $key The translation key
 * @param string $default Default translation
 */
function tcr_e($key, $default = '')
{
    echo tcr__($key, $default);
}

/**
 * Escape HTML and return the translation for the given key or use $default if
 * there is no translation
 *
 * @uses esc_html
 * @param string $key The translation key
 * @param string $default Default translation
 */
function tcr_esc_html__($key, $default = '')
{
    return esc_html(tcr__($key, $default));
}

/**
 * Escape for attributes and return the translation for the given key or use $default if
 * there is no translation.
 *
 * @uses esc_attr
 * @param string $key The translation key
 * @param string $default Default translation
 */
function tcr_esc_attr__($key, $default = '')
{
    return esc_attr(tcr__($key, $default));
}

/**
 * Escape HTML and echo the translation for the given key or use $default if
 * there is no translation
 *
 * @uses esc_html
 * @param string $key The translation key
 * @param string $default Default translation
 */
function tcr_esc_html_e($key, $default = '')
{
    echo tcr_esc_html__($key, $default);
}

/**
 * Escape for attributes and echo the translation for the given key or use $default if
 * there is no translation.
 *
 * @uses esc_attr
 * @param string $key The translation key
 * @param string $default Default translation
 */
function tcr_esc_attr_e($key, $default = '')
{
    echo tcr_esc_attr__($key, $default);
}

if (is_admin()) {
    // Load the admin (backend) functions
    require_once TCR_ADMIN_DIR . '/admin.php';
} else {
    // Load the theme (frontend) functions
    require_once TCR_INCLUDES_DIR . '/theme.php';
}