<?php

// Prevent direct script access
if (!defined('ABSPATH')) exit;

/**
 * Metabox options generator
 */
class MetaboxGenerator extends OptionsGenerator
{
    protected $config;
    protected $options;
    protected $prefix;

    public function __construct($config, $options)
    {
        $this->config = $config;
        $this->options = $options;
        $this->prefix = '_' . TCR_PREFIX . '_';

        add_action('add_meta_boxes', array(&$this, 'add'));
        add_action('save_post', array(&$this, 'save'));
    }

    public function add()
    {
        if (is_callable($this->config['callback'])) {
            $callback = $this->config['callback'];
        } else {
            $callback = array(&$this, 'render');
        }

        foreach ((array) $this->config['types'] as $type) {
            add_meta_box($this->config['id'], $this->config['title'], $callback, $type, $this->config['context'], $this->config['priority']);
        }
    }

    public function save($post_id)
    {
        // Bail if we're doing an auto save
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;

        // If our nonce isn't there, or we can't verify it, bail
        if (!isset($_POST["{$this->config['id']}_mb_nonce"]) || !wp_verify_nonce($_POST["{$this->config['id']}_mb_nonce"], "{$this->config['id']}_mb_save")) return;

        // If our current user can't edit this post, bail
        if ('page' == $_POST['post_type']) {
            if (!current_user_can('edit_page')) return;
        } else {
            if (!current_user_can('edit_post')) return;
        }

        // Only save if the current post type applies to our options
        if (!in_array($_POST['post_type'], (array) $this->config['types'])) return;

        foreach ($this->options as $option) {
            if (!empty($option['name'])) {
                $option['name'] = $this->prefix . $option['name'];
                if (isset($_POST[$option['name']])) {
                    switch ($option['type']) {
                        case 'toggle':
                            $value = 1;
                            break;
                        case 'upload':
                            $value = json_decode(stripslashes($_POST[$option['name']]), true);
                            break;
                        default:
                            $value = $_POST[$option['name']];
                            break;
                    }
                } else if ($option['type'] == 'toggle') {
                    $value = 0;
                } else {
                    $value = '';
                }

                $oldValue = get_post_meta($post_id, $option['name'], true);

                if ($value !== $oldValue) {
                    update_post_meta($post_id, $option['name'], $value);
                } else if ($value === '') {
                    delete_post_meta($post_id, $option['name'], $oldValue);
                }
            }
        }
    }

    public function render($post)
    {
        echo '<div class="tcr-meta-box-fields">';

        wp_nonce_field("{$this->config['id']}_mb_save", "{$this->config['id']}_mb_nonce");

        foreach ($this->options as $option) {
            $renderMethod = array_key_exists('render', $option) && method_exists($this, $option['render']) ? $option['render'] : $option['type'];

            if (method_exists($this, $renderMethod)) {
                $default = isset($option['default']) ? $option['default'] : '';
                $option['value'] = tcr_get_post_meta($post->ID, $option['name'], $default);
                $option['name'] = $this->prefix . $option['name'];
                $tooltip = isset($option['tooltip']) ? '<span class="tcr-tooltip-wrap"><span class="tcr-tooltip-content">' . $option['tooltip'] . '</span></span>' : '';

                echo '<div class="tcr-meta-box-field-outer tcr-clearfix">';
                echo '<div class="tcr-meta-box-field-label"><h4>' . $option['title'] . '</h4>' . $tooltip . '</div>';
                echo '<div class="tcr-meta-box-field-inner tcr-clearfix">';
                echo $this->{$renderMethod}($option);
                echo '</div>';
                if (isset($option['description']) && $option['description']) {
                    echo '<p class="tcr-meta-box-field-description">' . $option['description'] . '</p>';
                }
                echo '</div>';
            }

            if ($option['type'] == 'subtitle') {
                $classes = array();
                if (isset($option['class'])) {
                    $classes = explode(' ', $option['class']);
                }

                echo '<div class="tcr-meta-box-subtitle ' . join(' ', $classes) . '">' . esc_html($option['title']) . '</div>';
            }
        }

        echo '</div>';
    }
}