<?php

/**
 * admin.php
 *
 * Admin only functions
 */

// Prevent direct script access
if (!defined('ABSPATH')) exit;

require_once TCR_ADMIN_INCLUDES_DIR . '/OptionsGenerator.php';

global $pagenow;
if ('themes.php' == $pagenow && isset($_GET['activated'])) {
    // Activation
    tcr_activation();
}

/**
 * Theme activation function
 */
function tcr_activation()
{
    // Set the default options
    if (!is_array(get_option(TCR_OPTIONS_NAME))) {
        update_option(TCR_OPTIONS_NAME, tcr_get_default_options());
    }
}

add_action('after_setup_theme', 'tcr_admin_setup');


function tcr_admin_setup()
{
    // Set up translations
    load_theme_textdomain(TCR_ADMIN_TEXT_DOMAIN, TCR_ADMIN_DIR . '/languages');

}

add_action('admin_init', 'tcr_add_meta_boxes');

/**
 * Set up the meta boxes
 */
function tcr_add_meta_boxes()
{
    require_once TCR_ADMIN_INCLUDES_DIR . '/MetaboxGenerator.php';
    require_once TCR_ADMIN_INCLUDES_DIR . '/metaboxes.php';
}

add_action('admin_menu', 'tcr_admin_menu');

/**
 * Add the menu
 */
function tcr_admin_menu()
{
    add_theme_page(
        sprintf(__('%s Options', TCR_ADMIN_TEXT_DOMAIN), TCR_NAME),
        __('Theme Options', TCR_ADMIN_TEXT_DOMAIN),
        'edit_theme_options',
        'ThemeCatcher',
        'tcr_options_panel'
    );
}

/**
 * The options panel
 */
function tcr_options_panel()
{
    global $tcr;
    require_once TCR_ADMIN_INCLUDES_DIR . '/panel.php';
}

add_action('wp_ajax_tcr_save_options_ajax', 'tcr_save_options_ajax');

function tcr_save_options_ajax()
{
    if (isset($_POST['options']) && check_ajax_referer('tcr_save_options', false, false) && current_user_can('edit_theme_options')) {
        $options = json_decode(stripslashes($_POST['options']), true);

        $options = apply_filters('tcr_pre_save_options', $options);

        $options['last_saved'] = time();

        update_option(TCR_OPTIONS_NAME, $options);

        $response = array(
            'type' => 'success',
            'flush' => false
        );

        global $tcr;
        if ($options['portfolio_rewrite_slug'] && $tcr['options']['portfolio_rewrite_slug'] != $options['portfolio_rewrite_slug']) {
            $response['flush'] = true;
        }

        // Check if we need a custom stylesheet for Google font replacement or Custom CSS
        $customCss = array();
        if ($options['general_font'] != '') {
            $fonts = tcr_get_fonts();
            if (array_key_exists($options['general_font'], $fonts)) {
                $font = $fonts[$options['general_font']];
                if ($font['type'] === 'google') {
                    // User font selector
                    if ($options['general_font_selector'] != '') {
                        $customCss[] = $options['general_font_selector'] . ' { font-family: "' . $font['family'] . '", sans-serif; }';
                    }

                    // Built in selectors for options
                    $selectors = array();
                    if ($options['general_font_portfolio']) {
                        $selectors[] = '.portfolio-item-title';
                    }

                    if ($options['general_font_serene']) {
                        $selectors[] = '.serene-caption h1, .serene-caption h2, .serene-caption h3';
                    }

                    if ($options['general_font_fancybox']) {
                        $selectors[] = '#fancybox-title h1, #fancybox-title h2, #fancybox-title h3';
                    }

                    if (count($selectors)) {
                        $customCss[] = join(', ', $selectors) . ' { font-family: "' . $font['family'] . '", sans-serif; }';
                    }
                }
            }
        }

        if ($options['advanced_custom_css']) {
            $customCss[] = $options['advanced_custom_css'];
        }

        $customCssFile = TCR_CACHE_DIR . '/custom.css';
        if (is_writable(TCR_CACHE_DIR)) {
            ob_start();
            include TCR_ADMIN_INCLUDES_DIR . '/custom-css.php';
            @file_put_contents($customCssFile, ob_get_clean());
        }

        // Check if we need a custom JS file
        $customJsFile = TCR_CACHE_DIR . '/custom.js';
        if (is_writable(TCR_CACHE_DIR)) {
            if ($options['advanced_custom_js']) {
                ob_start(); ?>
jQuery(document).ready(function ($) {
    <?php echo $tcr['options']['advanced_custom_js'] . "\n"; ?>
});
                <?php
                @file_put_contents($customJsFile, ob_get_clean());
            } else if (file_exists($customJsFile)) {
                @unlink($customJsFile);
            }
        }

        header('Content-Type: application/json');
        echo json_encode($response);

        exit;
    }
}

add_filter('tcr_pre_save_options', 'tcr_pre_save_options');

/**
 * Sanitise the theme options before they are saved
 *
 * @param array $options
 */
function tcr_pre_save_options($options)
{
    // General options
    $options['general_search'] = isset($options['general_search']) ? true : false;
    $options['general_breadcrumbs'] = isset($options['general_breadcrumbs']) ? true : false;
    $options['general_sticky_header'] = isset($options['general_sticky_header']) ? true : false;
    $options['general_navigation_width'] = (int) $options['general_navigation_width'];
    $options['general_logo_top'] = (int) $options['general_logo_top'];
    $options['general_logo_left'] = (int) $options['general_logo_left'];
    $options['general_font_fullscreen'] = isset($options['general_font_fullscreen']) ? true : false;
    $options['general_font_portfolio'] = isset($options['general_font_portfolio']) ? true : false;
    $options['general_font_serene'] = isset($options['general_font_serene']) ? true : false;
    $options['general_font_fancybox'] = isset($options['general_font_fancybox']) ? true : false;

    // Background options
    $options['background_video_autostart'] = isset($options['background_video_autostart']) ? true : false;
    $options['background_video_start'] = (int) $options['background_video_start'];
    $options['background_audio_random'] = isset($options['background_audio_random']) ? true : false;
    $options['background_audio_autostart'] = isset($options['background_audio_autostart']) ? true : false;
    $options['background_speed'] = (int) $options['background_speed'];
    $options['background_fit_landscape'] = isset($options['background_fit_landscape']) ? true : false;
    $options['background_fit_portrait'] = isset($options['background_fit_portrait']) ? true : false;
    $options['background_fit_always'] = isset($options['background_fit_always']) ? true : false;
    $options['background_hide_speed'] = (int) $options['background_hide_speed'];
    $options['background_show_speed'] = (int) $options['background_show_speed'];
    $options['background_control_speed'] = (int) $options['background_control_speed'];
    $options['background_fadeIE'] = isset($options['background_fadeIE']) ? true : false;
    $options['background_save'] = isset($options['background_save']) ? true : false;
    $options['background_slideshow'] = isset($options['background_slideshow']) ? true : false;
    $options['background_slideshow_auto'] = isset($options['background_slideshow_auto']) ? true : false;
    $options['background_slideshow_speed'] = (int) $options['background_slideshow_speed'];
    $options['background_random'] = isset($options['background_random']) ? true : false;
    $options['background_keyboard'] = isset($options['background_keyboard']) ? true : false;
    $options['background_caption_speed'] = (int) $options['background_caption_speed'];
    $options['background_bullets'] = isset($options['background_bullets']) ? true : false;
    $options['background_low_quality'] = isset($options['background_low_quality']) ? true : false;
    $options['background_breaker'] = isset($options['background_breaker']) ? true : false;
    $options['background_breaker_on_max'] = isset($options['background_breaker_on_max']) ? true : false;

    // Blog options
    $options['blog_show_date_circle'] = isset($options['blog_show_date_circle']) ? true : false;
    $options['blog_single_featured_image'] = isset($options['blog_single_featured_image']) ? true : false;
    $options['blog_single_featured_float_width'] = (int) $options['blog_single_featured_float_width'];
    $options['blog_single_featured_float_height'] = (int) $options['blog_single_featured_float_height'];
    $options['blog_featured_image'] = isset($options['blog_featured_image']) ? true : false;
    $options['blog_fixed_featured_height'] = (int) $options['blog_fixed_featured_height'];
    $options['blog_show_author_description'] = isset($options['blog_show_author_description']) ? true : false;
    $options['blog_show_single_nav'] = isset($options['blog_show_single_nav']) ? true : false;
    $options['blog_featured_float_width'] = (int) $options['blog_featured_float_width'];
    $options['blog_featured_float_height'] = (int) $options['blog_featured_float_height'];

    // Portfolio options
    $options['portfolio_show_single_nav'] = isset($options['portfolio_show_single_nav']) ? true : false;
    $options['portfolio_featured_image'] = isset($options['portfolio_featured_image']) ? true : false;
    $options['portfolio_scale_featured_height'] = isset($options['portfolio_scale_featured_height']) ? true : false;
    $options['portfolio_fixed_featured_height'] = (int) $options['portfolio_fixed_featured_height'];
    $options['portfolio_video_height'] = (int) $options['portfolio_video_height'];
    $options['portfolio_video_width'] = (int) $options['portfolio_video_width'];
    $options['portfolio_comments'] = isset($options['portfolio_comments']) ? true : false;
    $options['portfolio_speed'] = (int) $options['portfolio_speed'];
    $options['portfolio_fit_landscape'] = isset($options['portfolio_fit_landscape']) ? true : false;
    $options['portfolio_fit_portrait'] = isset($options['portfolio_fit_portrait']) ? true : false;
    $options['portfolio_fit_always'] = isset($options['portfolio_fit_always']) ? true : false;
    $options['portfolio_control_speed'] = (int) $options['portfolio_control_speed'];
    $options['portfolio_fadeIE'] = isset($options['portfolio_fadeIE']) ? true : false;
    $options['portfolio_slideshow'] = isset($options['portfolio_slideshow']) ? true : false;
    $options['portfolio_slideshow_auto'] = isset($options['portfolio_slideshow_auto']) ? true : false;
    $options['portfolio_slideshow_speed'] = (int) $options['portfolio_slideshow_speed'];
    $options['portfolio_keyboard'] = isset($options['portfolio_keyboard']) ? true : false;
    $options['portfolio_caption_speed'] = (int) $options['portfolio_caption_speed'];
    $options['portfolio_bullets'] = isset($options['portfolio_bullets']) ? true : false;
    $options['portfolio_low_quality'] = isset($options['portfolio_low_quality']) ? true : false;

    // Footer options
    $options['footer_top_link'] = isset($options['footer_top_link']) ? true : false;

    return $options;
}

add_action('wp_ajax_tcr_export_options_ajax', 'tcr_export_options_ajax');

function tcr_export_options_ajax()
{
    if (isset($_POST['options']) && check_ajax_referer('tcr_export_options')) {
        $options = json_decode(stripslashes($_POST['options']), true);

        $options = apply_filters('tcr_pre_save_options', $options);

        $response = array(
            'type' => 'success',
            'data' => base64_encode(serialize($options))
        );

        header('Content-Type: application/json');
        echo json_encode($response);
    }

    exit;
}

add_action('wp_ajax_tcr_import_options_ajax', 'tcr_import_options_ajax');

function tcr_import_options_ajax()
{
    if (isset($_POST['options']) && check_ajax_referer('tcr_import_options')) {
        $options = @unserialize(base64_decode($_POST['options']));

        if (is_array($options)) {
            update_option(TCR_OPTIONS_NAME, $options);

            $response = array(
                'type' => 'success'
            );
        } else {
            $response = array(
                'type' => 'error',
                'message' => __('The import data is invalid', TCR_ADMIN_TEXT_DOMAIN)
            );
        }

        header('Content-Type: application/json');
        echo json_encode($response);
    }

    exit;
}

add_action('wp_ajax_tcr_reset_options_ajax', 'tcr_reset_options_ajax');

function tcr_reset_options_ajax()
{
    if (check_ajax_referer('tcr_reset_options')) {
        update_option(TCR_OPTIONS_NAME, tcr_get_default_options());

        $response = array(
            'type' => 'success'
        );

        header('Content-Type: application/json');
        echo json_encode($response);
    }

    exit;
}

add_action('admin_enqueue_scripts', 'tcr_enqueue_admin_styles', 10, 1);

function tcr_enqueue_admin_styles($hook)
{
    if (in_array($hook, array('appearance_page_ThemeCatcher', 'post.php', 'post-new.php', 'index.php', 'widgets.php'))) {
        wp_enqueue_style('tcr-admin', TCR_ADMIN_URL . '/css/styles.css', array(), TCR_VERSION);
    }
}

add_action('admin_enqueue_scripts', 'tcr_enqueue_admin_scripts', 10, 1);

function tcr_enqueue_admin_scripts($hook)
{
    if (in_array($hook, array('appearance_page_ThemeCatcher', 'post.php', 'post-new.php', 'index.php', 'widgets.php'))) {
        wp_register_script('tcr-plugins', TCR_ADMIN_URL . '/js/plugins.js', array('jquery'), TCR_VERSION, true);
        wp_register_script('tcr-admin', TCR_ADMIN_URL . '/js/admin.js', array('jquery', 'json2'), TCR_VERSION, true);
        wp_register_script('tcr-shortcodes', TCR_ADMIN_URL . '/js/shortcodes.js', array('jquery', 'tcr-plugins', 'tcr-admin'), TCR_VERSION, true);

        if ($hook === 'appearance_page_ThemeCatcher') {
            wp_register_script('tcr-panel', TCR_ADMIN_URL . '/js/panel.js', array('jquery', 'swfupload-all', 'jquery-ui-sortable', 'tcr-plugins', 'tcr-admin'), TCR_VERSION, true);
            wp_enqueue_script('tcr-panel');
        } elseif (in_array($hook, array('post.php', 'post-new.php'))) {
            wp_register_script('tcr-metaboxes', TCR_ADMIN_URL . '/js/metaboxes.js', array('jquery', 'swfupload-all', 'jquery-ui-sortable', 'tcr-plugins'), TCR_VERSION, true);
            wp_enqueue_script('tcr-metaboxes');
            wp_enqueue_script('tcr-shortcodes');
        } elseif ($hook === 'index.php' || $hook === 'widgets.php') {
            wp_enqueue_script('tcr-shortcodes');
        }

        wp_localize_script('tcr-admin', 'tcrL10n', array(
            'siteUrl' => site_url(),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'themeUrl' => TCR_URL,
            'themeAdminUrl' => TCR_ADMIN_URL,
            'swfuploadFlashUrl' => includes_url('js/swfupload/swfupload.swf'),
            'swfuploadUploadUrl' => admin_url(),
            'authCookie' => is_ssl() ? $_COOKIE[SECURE_AUTH_COOKIE] : $_COOKIE[AUTH_COOKIE],
            'loggedInCookie' => $_COOKIE[LOGGED_IN_COOKIE],
            'timthumbUrl' => TCR_URL . '/includes/timthumb.php',
            'swfuploadNonce' => wp_create_nonce('tcr_swfupload'),
            'deleteImage' => __('Delete image', TCR_ADMIN_TEXT_DOMAIN),
            'editSettings' => __('Edit settings', TCR_ADMIN_TEXT_DOMAIN),
            'deleteTrack' => __('Delete track', TCR_ADMIN_TEXT_DOMAIN),
            'editTrack' => __('Edit track', TCR_ADMIN_TEXT_DOMAIN),
            'moveTrack' => __('Move track', TCR_ADMIN_TEXT_DOMAIN),
            'untitled' => __('Untitled', TCR_ADMIN_TEXT_DOMAIN),
        	'upload' => __('Upload', TCR_ADMIN_TEXT_DOMAIN),
            'uploadTooManyFiles' => __('You have attempted to upload too many files', TCR_ADMIN_TEXT_DOMAIN),
            'imageFiles' => __('Image Files', TCR_ADMIN_TEXT_DOMAIN),
            'm4aFileUrl' => __('M4A file URL', TCR_ADMIN_TEXT_DOMAIN),
            'mp3FileUrl' => __('MP3 file URL', TCR_ADMIN_TEXT_DOMAIN),
            'oggFileUrl' => __('OGG file URL (optional)', TCR_ADMIN_TEXT_DOMAIN),
            'on' => __('On', TCR_ADMIN_TEXT_DOMAIN),
            'off' => __('Off', TCR_ADMIN_TEXT_DOMAIN),
            '_default' => __('Default', TCR_ADMIN_TEXT_DOMAIN),
            'yes' => __('Yes', TCR_ADMIN_TEXT_DOMAIN),
            'no' => __('No', TCR_ADMIN_TEXT_DOMAIN),
            'name' => __('Name', TCR_ADMIN_TEXT_DOMAIN),
            'saveNonce' => wp_create_nonce('tcr_save_options'),
            'resetNonce' => wp_create_nonce('tcr_reset_options'),
            'importNonce' => wp_create_nonce('tcr_import_options'),
            'exportNonce' => wp_create_nonce('tcr_export_options'),
            'confirmResetTranslations' => __('Are you sure you want to reset all translations?', TCR_ADMIN_TEXT_DOMAIN),
            'shortcodePreviewUrl' => admin_url('?tcr_shortcode_preview=1'),
            'insert' => __('Insert', TCR_ADMIN_TEXT_DOMAIN),
            'insert_close' => __('Insert & Close', TCR_ADMIN_TEXT_DOMAIN),
            'close' => __('Close', TCR_ADMIN_TEXT_DOMAIN),
            'save' => __('Save', TCR_ADMIN_TEXT_DOMAIN),
            'cancel' => __('Cancel', TCR_ADMIN_TEXT_DOMAIN),
            'title' => __('Title', TCR_ADMIN_TEXT_DOMAIN),
            'content' => __('Content', TCR_ADMIN_TEXT_DOMAIN),
            'removeTab' => __('Remove tab', TCR_ADMIN_TEXT_DOMAIN),
            'removeToggle' => __('Remove toggle', TCR_ADMIN_TEXT_DOMAIN),
            'removeSlide' => __('Remove slide', TCR_ADMIN_TEXT_DOMAIN),
            'flushingRewriteRules' => __('Flushing rewrite rules', TCR_ADMIN_TEXT_DOMAIN),
            'reloadingPage' => __('Reloading page', TCR_ADMIN_TEXT_DOMAIN),
            'startingImport' => __('Starting import', TCR_ADMIN_TEXT_DOMAIN),
            'confirmResetOptions' => __('Are you sure you want to reset all options?', TCR_ADMIN_TEXT_DOMAIN),
            'px' => _x('px', 'pixels', TCR_ADMIN_TEXT_DOMAIN)
        ));
    }
}

add_action('media_buttons', 'tcr_insert_shortcode_button', 20, 0);

/**
 * Add the "Insert shortcode" button to the end of the media buttons above a post/page
 *
 * @param string $widgetId The ID of the text widget textarea
 */
function tcr_insert_shortcode_button($widgetId = '')
{
    $href = admin_url('admin-ajax.php?action=tcr_insert_shortcode');
    if ($widgetId) {
        $href .= '&widget=' . urlencode($widgetId);
    }
    echo '<a href="' . $href . '" class="tcr-shortcode-trigger" title="' . esc_attr__('Insert a shortcode', TCR_ADMIN_TEXT_DOMAIN) . '"><img src="' . TCR_ADMIN_URL . '/images/insert-shortcode.png' . '" alt="' . esc_attr__('Insert a shortcode', TCR_ADMIN_TEXT_DOMAIN) . '" /></a>';
}

add_action('wp_ajax_tcr_insert_shortcode', 'tcr_insert_shortcode');

/**
 * The "Insert shortcode" dialog
 */
function tcr_insert_shortcode()
{
    global $tcr;
    require_once TCR_ADMIN_INCLUDES_DIR . '/ShortcodeOptionsGenerator.php';
    require_once TCR_ADMIN_INCLUDES_DIR . '/shortcodes.php';
    exit;
}

add_action('wp_ajax_nopriv_tcr_insert_shortcode', 'tcr_insert_shortcode_nopriv');

function tcr_insert_shortcode_nopriv()
{
    wp_die(__('You do not have permission to perform this action. Are you logged in?', TCR_ADMIN_TEXT_DOMAIN));
}

function tcr_background_settings_form()
{
    ?><div id="tcr-background-settings-overlay" class="tcr-lightbox-overlay">
    <div id="tcr-background-settings" class="tcr-lightbox">
        <table>
            <tr>
                <th scope="row"><label for="tcr-background-caption"><?php esc_html_e('Caption HTML', TCR_ADMIN_TEXT_DOMAIN); ?></label></th>
                <td>
                    <textarea id="tcr-background-caption" name="tcr-background-caption"></textarea>
                    <p class="tcr-description"><?php esc_html_e('Enter the caption HTML, below is an example.', TCR_ADMIN_TEXT_DOMAIN); ?></p>
                    <pre>&lt;h1&gt;<?php esc_html_e('Sample Title', TCR_ADMIN_TEXT_DOMAIN); ?>&lt;/h1&gt;
&lt;p&gt;<?php esc_html_e('Sample description.', TCR_ADMIN_TEXT_DOMAIN); ?>&lt;/p&gt;</pre>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="tcr-background-caption-position"><?php esc_html_e('Caption position', TCR_ADMIN_TEXT_DOMAIN); ?></label>
                </th>
                <td>
                    <select id="tcr-background-caption-position" name="tcr-background-caption-position">
                        <option value=""><?php esc_html_e('Default', TCR_ADMIN_TEXT_DOMAIN); ?></option>
                        <option value="random"><?php esc_html_e('Random', TCR_ADMIN_TEXT_DOMAIN); ?></option>
                        <option value="left top"><?php esc_html_e('left top', TCR_ADMIN_TEXT_DOMAIN); ?></option>
                        <option value="left center"><?php esc_html_e('left center', TCR_ADMIN_TEXT_DOMAIN); ?></option>
                        <option value="left bottom"><?php esc_html_e('left bottom', TCR_ADMIN_TEXT_DOMAIN); ?></option>
                        <option value="center top"><?php esc_html_e('center top', TCR_ADMIN_TEXT_DOMAIN); ?></option>
                        <option value="center center"><?php esc_html_e('center center', TCR_ADMIN_TEXT_DOMAIN); ?></option>
                        <option value="center bottom"><?php esc_html_e('center bottom', TCR_ADMIN_TEXT_DOMAIN); ?></option>
                        <option value="right top"><?php esc_html_e('right top', TCR_ADMIN_TEXT_DOMAIN); ?></option>
                        <option value="right center"><?php esc_html_e('right center', TCR_ADMIN_TEXT_DOMAIN); ?></option>
                        <option value="right bottom"><?php esc_html_e('right bottom', TCR_ADMIN_TEXT_DOMAIN); ?></option>
                    </select>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="tcr-background-title"><?php esc_html_e('Title', TCR_ADMIN_TEXT_DOMAIN); ?></label>
                </th>
                <td>
                    <input type="text" id="tcr-background-title" name="tcr-background-title" />
                    <p class="tcr-description"><?php esc_html_e('This is shown when hovering the bullet navigation in full screen mode', TCR_ADMIN_TEXT_DOMAIN); ?></p>
                </td>
            </tr>
        </table>
        <div class="tcr-clearfix">
            <a id="tcr-background-caption-save" class="tcr-button tcr-blue"><?php esc_html_e('Save', TCR_ADMIN_TEXT_DOMAIN); ?></a>
            <a id="tcr-background-caption-cancel" class="tcr-button tcr-light"><?php esc_html_e('Cancel', TCR_ADMIN_TEXT_DOMAIN); ?></a>
        </div>
    </div>
</div><?php
}

add_action('wp_ajax_tcr_flush_rewrite_rules_ajax', 'tcr_flush_rewrite_rules_ajax');

function tcr_flush_rewrite_rules_ajax()
{
    flush_rewrite_rules();
    exit;
}


add_filter('manage_edit-portfolio_columns', 'edit_portfolio_columns');

/**
 * Add the portfolio post type columns to the list
 */
function edit_portfolio_columns($columns)
{
    $columns['portfolio_categories'] = __('Categories', TCR_ADMIN_TEXT_DOMAIN);
	$columns['description'] = __('Description', TCR_ADMIN_TEXT_DOMAIN);
	$columns['thumbnail'] = __('Thumbnail', TCR_ADMIN_TEXT_DOMAIN);
	return $columns;
}

add_action('manage_posts_custom_column', 'manage_portfolio_columns');

/**
 * Display the portfolio post type column values
 */
function manage_portfolio_columns($column) {
	global $post;

    if ($post->post_type == 'portfolio') {
		switch($column){
			case 'description':
				the_excerpt();
				break;
			case 'portfolio_categories':
				$terms = get_the_terms($post->ID, 'portfolio_category');

				if ($terms) {
				    $output = array();
				    foreach($terms as $term) {
				        $output[] = esc_html(sanitize_term_field('name', $term->name, $term->term_id, 'portfolio_category', 'display'));
				    }
				    echo join(', ', $output);
				}
				break;
			case 'thumbnail':
			    if (has_post_thumbnail()) {
				    echo tcr_get_image_tag(tcr_get_image_src(get_post_thumbnail_id(), array('width' => 150, 'height' => 84)));
			    }
				break;
		}
	}
}

function tcr_get_background_overlay_options($showDefaultOption = false)
{
    $options = array(
        'diagonal' => __('Diagonal', TCR_ADMIN_TEXT_DOMAIN),
        'diagonal-rev' => __('Diagonal Reverse', TCR_ADMIN_TEXT_DOMAIN),
        'wave' => __('Wave', TCR_ADMIN_TEXT_DOMAIN),
        'scales' => __('Scales', TCR_ADMIN_TEXT_DOMAIN),
        'mosaic' => __('Mosaic', TCR_ADMIN_TEXT_DOMAIN),
        'checked' => __('Checked', TCR_ADMIN_TEXT_DOMAIN),
        'cross' => __('Cross', TCR_ADMIN_TEXT_DOMAIN),
        'cross2' => __('Cross 2', TCR_ADMIN_TEXT_DOMAIN),
        'square' => __('Square', TCR_ADMIN_TEXT_DOMAIN),
        'diagonal-dash' => __('Diagonal Dash', TCR_ADMIN_TEXT_DOMAIN),
        'diagonal-dash-rev' => __('Diagonal Dash Reverse', TCR_ADMIN_TEXT_DOMAIN),
        'dot' => __('Dot', TCR_ADMIN_TEXT_DOMAIN),
        'vert-thin' => __('Vertical line thin', TCR_ADMIN_TEXT_DOMAIN),
        'vert-thick' => __('Vertical line thick', TCR_ADMIN_TEXT_DOMAIN),
        'horz-thin' => __('Horizontal line thin', TCR_ADMIN_TEXT_DOMAIN),
        'horz-thick' => __('Horizontal line thick', TCR_ADMIN_TEXT_DOMAIN),
        'none' => __('None', TCR_ADMIN_TEXT_DOMAIN)
    );

    if ($showDefaultOption) {
        $options = array('' => __('Default', TCR_ADMIN_TEXT_DOMAIN)) + $options;
    }

    return $options;
}

add_action('after_setup_theme', 'tcr_swfupload_set_auth_cookies');

/**
 * Sets the correct authentication cookies when uploading files via SWFUpload
 */
function tcr_swfupload_set_auth_cookies()
{
    if (isset($_POST['tcr_swfupload']) && $_POST['tcr_swfupload'] == 1) {
        // Flash often fails to send cookies with the POST or upload, so we need to pass it in GET or POST instead
        if (is_ssl() && empty($_COOKIE[SECURE_AUTH_COOKIE]) && !empty($_REQUEST['auth_cookie'])) {
            $_COOKIE[SECURE_AUTH_COOKIE] = $_REQUEST['auth_cookie'];
        } elseif (empty($_COOKIE[AUTH_COOKIE]) && !empty($_REQUEST['auth_cookie'])) {
            $_COOKIE[AUTH_COOKIE] = $_REQUEST['auth_cookie'];
        }

        if (empty($_COOKIE[LOGGED_IN_COOKIE]) && !empty($_REQUEST['logged_in_cookie'])) {
            $_COOKIE[LOGGED_IN_COOKIE] = $_REQUEST['logged_in_cookie'];
        }
    }
}

add_action('admin_init', 'tcr_swfupload_upload_ajax');

function tcr_swfupload_upload_ajax()
{
    if (isset($_POST['tcr_swfupload'], $_POST['action'], $_POST['_wpnonce']) && $_POST['tcr_swfupload'] == 1 && $_POST['action'] == 'tcr_swfupload_upload_ajax') {
        if (current_user_can('upload_files') && wp_verify_nonce($_POST['_wpnonce'], 'tcr_swfupload')) {
            $result = wp_handle_upload($_FILES['upload'], array('action' => $_POST['action']));

            if (is_array($result)) {
                if (array_key_exists('error', $result)) {
                    $response = array(
                        'type' => 'error',
                        'message' => $result['error']
                    );
                } else {
                    // Remove the site url from the image url
                    $result['url'] = str_replace(site_url(), '', $result['url']);
                    $result['file'] = str_replace(ABSPATH, '', $result['file']);

                    $response = array(
                        'type' => 'success',
                        'data' => $result
                    );
                }
            } else {
                $response = array(
                    'type' => 'error',
                    'message' => __('An unknown file upload error occurred', TCR_ADMIN_TEXT_DOMAIN)
                );
            }

            header('Content-Type: application/json');
            echo json_encode($response);
        }

        // Prevent the rest of WP loading
        exit;
    }
}

add_action('admin_init', 'tcr_swfupload_upload_audio_ajax');

function tcr_swfupload_upload_audio_ajax()
{
    if (isset($_POST['tcr_swfupload'], $_POST['action'], $_POST['_wpnonce']) && $_POST['tcr_swfupload'] == 1 && $_POST['action'] == 'tcr_swfupload_upload_audio_ajax') {
        if (current_user_can('upload_files') && wp_verify_nonce($_POST['_wpnonce'], 'tcr_swfupload')) {
            $result = wp_handle_upload($_FILES['upload'], array('action' => $_POST['action']));

            if (is_array($result)) {
                if (array_key_exists('error', $result)) {
                    $response = array(
                        'type' => 'error',
                        'message' => $result['error']
                    );
                } else {
                    $response = array(
                        'type' => 'success',
                        'data' => $result
                    );
                }
            } else {
                $response = array(
                    'type' => 'error',
                    'message' => __('An unknown file upload error occurred', TCR_ADMIN_TEXT_DOMAIN)
                );
            }

            header('Content-Type: application/json');
            echo json_encode($response);
        }

        // Prevent the rest of WP loading
        exit;
    }
}

add_action('wp_ajax_tcr_delete_file_ajax', 'tcr_delete_file_ajax');

function tcr_delete_file_ajax()
{
    if (current_user_can('upload_files') && isset($_GET['file']) && strlen($_GET['file'])) {
        $file = path_join(ABSPATH, $_GET['file']);

        if (file_exists($file)) {
            @unlink($file);
        }
    }
    exit;
}

/**
 * Returns an array of all posts in the format:
 *
 * id => title
 *
 * @return array
 */
function tcr_get_all_post_options()
{
	$posts = get_posts(array('post_type' => 'post', 'numberposts' => 0));
	$options = array();

	foreach($posts as $post) {
		$options[$post->ID] = $post->post_title;
	}

	return $options;
}

/**
 * Returns an array of all categories in the format:
 *
 * id => name
 *
 * @return array
 */
function tcr_get_all_category_options()
{
	$categories = get_terms('category', array('hide_empty' => false));
	$options = array();

	foreach($categories as $category) {
		$options[$category->term_id] = $category->name;
	}

	return $options;
}

/**
 * Returns an array of all authors in the format:
 *
 * id => display_name
 *
 * @return array
 */
function tcr_get_all_author_options()
{
	global $wpdb;
	$options = array();
	$authors = $wpdb->get_results("SELECT ID, display_name FROM $wpdb->users WHERE ID IN (SELECT post_author FROM $wpdb->posts) ORDER BY display_name");

	foreach ($authors as $author) {
		$options[$author->ID] = $author->display_name;
	}

	return $options;
}

/**
 * Returns an array of all background groups
 *
 * id => group_name
 *
 * @return array
 */
function tcr_get_background_group_options()
{
    global $tcr;
    $options = array(
        ''  => __('Select a group...', TCR_ADMIN_TEXT_DOMAIN)
    );

    foreach ($tcr['options']['background_groups'] as $groupId => $group) {
        $options[$groupId] = $group['name'];
    }

    return $options;
}

add_action('auth_redirect', 'tcr_shortcode_preview');

function tcr_shortcode_preview()
{
    if (isset($_GET['tcr_shortcode_preview']) && $_GET['tcr_shortcode_preview'] == '1') {
        require_once TCR_ADMIN_INCLUDES_DIR . '/shortcode-preview.php';
        exit;
    }
}

function tcr_shortcode_preview_html()
{
    ?>
    <div class="tcr-sc-preview-outer">
        <div class="tcr-sc-preview-label"><h4><?php esc_html_e('Preview', TCR_ADMIN_TEXT_DOMAIN); ?></h4><span class="tcr-sc-preview-refresh" title="<?php esc_attr_e('Refresh preview', TCR_ADMIN_TEXT_DOMAIN); ?>"></span></div>
        <div class="tcr-sc-preview-inner"><iframe class="tcr-sc-preview-content-frame" frameBorder="0"></iframe></div>
    </div>
    <?php
}